package de.renew.engine.simulator;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.junit.Before;
import org.junit.Test;

import de.renew.util.Semaphor;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.atLeastOnce;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;


/**
 * @author Felix Ortmann
 *
 * TestCase for the {@link WrappedFutureTask}.
 *
 */
public class TestWrappedFutureTask {
    private BlockingSimulationCallable<Object> _sampleBlock1;
    private BlockingSimulationCallable<String> _sampleBlock2;
    private BlockingSimulationCallable<Object> _sampleBlock3;
    private Semaphor _lock;
    private Callable<Object> _sampleCallable1;
    private Callable<String> _sampleCallable2;
    private Callable<Object> _sampleCallable3;
    private Object _returnValue1;
    private String _returnValue2;
    private WrappedFutureTask<Object> _underTest1;
    private WrappedFutureTask<String> _underTest2;
    private WrappedFutureTask<Object> _underTest3;

    private static final String TEST_STRING = "some String to be returned...";

    /**
     */
    @Before
    public void setUp() {
        _returnValue1 = new Object();
        _returnValue2 = TEST_STRING;

        _sampleCallable1 = () -> {
            //we do something and then return what we computed
            return _returnValue1;
        };

        _sampleCallable2 = () -> {
            //we do something and then return what we computed
            return _returnValue2;
        };

        _sampleCallable3 = () -> {
            //we do something and then return what we computed
            //here we wait a bit ;)
            Thread.sleep(2000); //2 seconds
            return _returnValue2; //strings are also objects
        };

        _lock = new Semaphor();

        _sampleBlock1 =
            new BlockingSimulationCallable<>(_sampleCallable1, _lock, Thread.currentThread());
        _sampleBlock2 =
            new BlockingSimulationCallable<>(_sampleCallable2, _lock, Thread.currentThread());
        _sampleBlock3 =
            new BlockingSimulationCallable<>(_sampleCallable3, _lock, Thread.currentThread());

        _underTest1 = new WrappedFutureTask<>(_sampleBlock1);
        _underTest2 = new WrappedFutureTask<>(_sampleBlock2);
        _underTest3 = new WrappedFutureTask<>(_sampleBlock3);
    }

    /**
     * Test method for {@link de.renew.engine.simulator.WrappedFutureTask#get()}.
     */
    @Test
    public final void testGetNormal() {
        //first we assume that everything is ok, we just want the return values.
        System.out.println("Execute the test tasks to be able to use the get() method.");
        SimulationThreadPool pool = SimulationThreadPool.getNew();
        pool.execute(_underTest1);
        pool.execute(_underTest2);
        pool.execute(_underTest3);
        try {
            System.out.println(
                "After this line there should no exception be thrown! \nTrying to get the return values.");
            assertEquals(_returnValue1, _underTest1.get());
            assertEquals(_returnValue2, _underTest2.get());
            assertEquals(_returnValue2, _underTest3.get());
        } catch (InterruptedException | ExecutionException e) {
            e.printStackTrace();
        }
    }

    /**
     * Test method for {@link de.renew.engine.simulator.WrappedFutureTask#get()}.
     */
    @Test
    public final void testGetWithInterrupt() {
        // now we voluntarily enforce an interrupt, to check the deblocking function of
        // get in case of exceptions.
        _sampleCallable1 = () -> {
            throw new InterruptedException();
        };

        _sampleCallable2 = () -> {
            throw new ExecutionException(null);
        };

        //now we need semaphore mocks to test if methods were invoked on them
        _lock = mock(Semaphor.class);

        //wrap faulty callables into the blockingCallables
        _sampleBlock1 =
            new BlockingSimulationCallable<>(_sampleBlock1, _lock, Thread.currentThread());
        _sampleBlock2 =
            new BlockingSimulationCallable<>(_sampleBlock2, _lock, Thread.currentThread());

        //create new WrappedFutureTasks to test their behaviour under exceptions...
        _underTest1 = new WrappedFutureTask<>(_sampleBlock1);
        _underTest2 = new WrappedFutureTask<>(_sampleBlock2);

        System.out.println("Execute the test tasks to be able to use the get() method.");
        SimulationThreadPool pool = SimulationThreadPool.getNew();
        pool.execute(_underTest1);
        pool.execute(_underTest2);
        pool.execute(_underTest3);

        try {
            System.out.println(
                "After this line an exception should be thrown, invisible on the console.");
            _underTest1.get();
        } catch (InterruptedException | ExecutionException e) {
            // when we got here, the wrappedFutureTask exception handling should have made
            // sure, that the lock is unblocked by now (V() was called). This is what we test for now.
            verify(_lock, atLeastOnce()).V(); //at least once the semaphore was unlocked.
                                              //this has to be, cause wrappedFutureTask shall unlock in case of exception!
        }

        try {
            System.out.println(
                "After this line an exception should be thrown, invisible on the console.");
            _underTest2.get();
        } catch (InterruptedException | ExecutionException e) {
            // when we got here, the wrappedFutureTask exception handling should have made
            // sure, that the lock is unblocked by now (V() was called). This is what we test for now.
            verify(_lock, atLeastOnce()).V(); //at least once the semaphore was unlocked.
                                              //this has to be, cause wrappedFutureTask shall unlock in case of exception!
        }
    }

    /**
     * Test method for {@link de.renew.engine.simulator.WrappedFutureTask#get(long, java.util.concurrent.TimeUnit)}.
     */
    @Test
    public final void testGetLongTimeUnit() {
        //now we wait a given amount of time for the result to be computed.
        //first we assume that everything is ok, we just want the return values.
        System.out.println("Execute the test tasks to be able to use the get() method.");
        SimulationThreadPool pool = SimulationThreadPool.getNew();
        pool.execute(_underTest1);
        pool.execute(_underTest2);
        pool.execute(_underTest3);
        try {
            System.out.println(
                "After this line there should no exception be thrown! \nTrying to get the return values.");
            assertEquals(_returnValue1, _underTest1.get(1000, TimeUnit.MILLISECONDS)); //max. wait a sec.
            assertEquals(_returnValue2, _underTest2.get(1000, TimeUnit.MILLISECONDS)); //max. wait a sec.
            assertEquals(_returnValue2, _underTest3.get(2500, TimeUnit.MILLISECONDS)); //max. wait 2.5 secs.
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            e.printStackTrace();
        }
    }

    /**
     * Test method for {@link de.renew.engine.simulator.WrappedFutureTask#get(long, java.util.concurrent.TimeUnit)}.
     */
    @Test
    public final void testGetLongTimeUnitWithTimeout() {
        //now we wait a given amount of time for the result to be computed.

        //first set up a mock semaphore
        _lock = mock(Semaphor.class);

        _sampleBlock3 =
            new BlockingSimulationCallable<>(_sampleCallable3, _lock, Thread.currentThread());
        _underTest3 = new WrappedFutureTask<>(_sampleBlock3);

        System.out.println("Execute the test tasks to be able to use the get() method.");
        SimulationThreadPool pool = SimulationThreadPool.getNew();
        pool.execute(_underTest1);
        pool.execute(_underTest2);
        pool.execute(_underTest3);

        try {
            System.out.println(
                "After this line an exception should be thrown, which is invisible on the console.");
            System.out.println("If actually a visible exception is thrown, the test has failed!");
            //now we wait too short cause this callable needs at least 2 seconds to finish.
            _underTest3.get(1000, TimeUnit.MILLISECONDS); //max. wait a sec.
        } catch (InterruptedException | ExecutionException e) {
            e.printStackTrace();
            //if we break here, the test fails - see stacktrace
        } catch (TimeoutException e) {
            verify(_lock, atLeastOnce()).V(); // since we timed out, we want to verify the
                                              //exception handling of wrappedFutureTask and test, whether the semaphore is unlocked now.
        }
    }
}