package de.renew.engine.simulator;

import java.util.concurrent.Callable;

import org.junit.Before;
import org.junit.Test;

import de.renew.util.Semaphor;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.mockito.Mockito.atLeast;
import static org.mockito.Mockito.atLeastOnce;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;


/**
 *
 * @author Felix Ortmann
 *
 * TestCase for the {@link BlockingSimulationCallable}
 *
 */
public class TestBlockingSimulationCallable {
    private BlockingSimulationCallable<Object> _underTest1;
    private BlockingSimulationCallable<String> _underTest2;
    private BlockingSimulationCallable<?> _underTest3;
    private SimulationThreadPool _pool;
    private Object _returnValueObject;
    private String _returnValueString;
    private Semaphor _lock;
    private SimulationThread _thread;
    private Runnable _run;

    private static final String TEST_THREAD_NAME = "testThread";
    private static final String TEST_STRING = "a String to experiment with";

    @Before
    public void setUp() {
        _returnValueObject = new Object();
        _returnValueString = TEST_STRING;

        _pool = SimulationThreadPool.getNew();

        _lock = mock(Semaphor.class);
        _run = () -> {
            try {
                assertEquals(_returnValueObject, _underTest1.call());
            } catch (Exception e) {
                e.printStackTrace();
            }
        };

        _thread = new SimulationThread(
            Thread.currentThread().getThreadGroup(), _run, TEST_THREAD_NAME, 5);

        //here we do compute something, what we want to return via the call method
        Callable<Object> innerCallable1 = () -> {
            //here we do compute something, what we want to return via the call method
            return _returnValueObject;
        };

        _underTest1 = new BlockingSimulationCallable<>(innerCallable1, _lock, _thread);

        //here we do compute something, what we want to return via the call method
        Callable<String> innerCallable2 = () -> {
            //here we do compute something, what we want to return via the call method
            return _returnValueString;
        };

        _underTest2 = new BlockingSimulationCallable<>(innerCallable2, _lock, _thread);

        _underTest3 = new BlockingSimulationCallable<>(innerCallable1, _lock, null);
    }

    @Test
    public final void testGetAncestor() {
        assertEquals(_thread, _underTest1.getAncestor());
        assertEquals(_thread, _underTest2.getAncestor());
        assertNull(_underTest3.getAncestor());
    }

    @Test
    public final synchronized void testCall() {
        System.out.println(
            "Trying to get the values by calling the inner callables call() methods. No Exception should be thrown here.");

        //for the first test, everything is prepared from the setUp()
        _pool.execute(_thread);

        _run = () -> {
            try {
                assertEquals(_returnValueString, _underTest2.call());
            } catch (Exception e) {
                e.printStackTrace();
            }
        };

        //prepare for testing the second callable
        _thread = new SimulationThread(
            Thread.currentThread().getThreadGroup(), _run, TEST_THREAD_NAME, 5);

        _pool.execute(_thread);

        _run = () -> {
            try {
                assertEquals(_returnValueObject, _underTest3.call());
            } catch (Exception e) {
                e.printStackTrace();
            }
        };

        //prepare for testing the second callable
        _thread = new SimulationThread(
            Thread.currentThread().getThreadGroup(), _run, TEST_THREAD_NAME, 5);

        _pool.execute(_thread);
        //even in the case of an exception, the finally part of the callables call method 
        //should have been executed at least three times (this is the abort in the semaphore)

        //since the pool executes the threads concurrently we want it to have enough time
        try {
            System.out.println("Waiting for SimulationThreadPool to execute.");
            wait(1500);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        verify(_lock, atLeast(3)).V();
    }

    @Test
    public final void testAbort() {
        _thread.setAncestor(Thread.currentThread());
        assertNotNull(_thread.getAncestor());
        _underTest1.abort(_thread);
        verify(_lock, atLeastOnce()).V(); // the abort method should unlock the semaphore
        assertNull(_thread.getAncestor());
    }
}