package de.renew.net;

import java.io.Serializable;
import java.util.NoSuchElementException;


/**
 * A TimeSet collects a set of doubles interpreted as
 * points of time. The handling is done by an AVL tree,
 * which handles all requests in O(log n) time.
 */
public class TimeSet implements Serializable {
    /** A special singleton object that denotes the empty set.*/
    public static final TimeSet EMPTY = new TimeSet(null, 0, 0, null);

    /**
     * A special singleton object that denotes the
     * set with a single object 0.
     */
    public static final TimeSet ZERO = EMPTY.including(0.0);

    // All instance variables are supposed to be final,
    // but they must not be declared that way due to a compiler bug.

    /** The left subtree which handles the earlier events. */
    private TimeSet left;

    /** The local time of the {@code TimeSet}. */
    private double time;

    /** The right subtree which handles the later events. */
    private TimeSet right;

    /**
     * The number of times that the time of this {@code TimeSet} is contained in the
     * set. This is useful if there are time instances when lots of events happen at once.
     * This is especially important for untimed situations.
     */
    private int mult;

    /** The height of this {@code TimeSet}. It is used to balance the tree. */
    private int height;

    /** The size of this {@code TimeSet}, which may be queried. */
    private int size;

    /** The size of this {@code TimeSet} without counting duplicates. It may be queried. */
    private int uniqueSize;

    /**
     * This constructor of a tree node takes care of rotating
     * other tree nodes as required. No modifications are
     * performed for other tree nodes.
     */
    private TimeSet(TimeSet left, double time, int mult, TimeSet right) {
        if (mult == 0) {
            // This is the special empty set.
            if (left != null || right != null || time != 0) {
                throw new RuntimeException("Bad empty time set parameters.");
            }
            this.left = null;
            this.right = null;
            this.mult = 0;
            this.time = 0;
            height = 0;
            size = 0;
            uniqueSize = 0;
        } else {
            // This is an ordinary time set.
            int leftHeight = left.height;
            int rightHeight = right.height;
            int diff = leftHeight - rightHeight;
            if (diff >= 2) {
                // Left side too high.
                int leftLeftHeight = left.left.height;
                int leftRightHeight = left.right.height;
                if (leftLeftHeight > leftRightHeight) {
                    this.left = left.left;
                    this.time = left.time;
                    this.mult = left.mult;
                    this.right = new TimeSet(left.right, time, mult, right);
                } else {
                    this.left = new TimeSet(left.left, left.time, left.mult, left.right.left);
                    this.time = left.right.time;
                    this.mult = left.right.mult;
                    this.right = new TimeSet(left.right.right, time, mult, right);
                }
            } else if (diff <= -2) {
                // Right side too high.
                int rightRightHeight = right.right.height;
                int rightLeftHeight = right.left.height;
                if (rightRightHeight > rightLeftHeight) {
                    this.left = new TimeSet(left, time, mult, right.left);
                    this.time = right.time;
                    this.mult = right.mult;
                    this.right = right.right;
                } else {
                    this.left = new TimeSet(left, time, mult, right.left.left);
                    this.time = right.left.time;
                    this.mult = right.left.mult;
                    this.right = new TimeSet(right.left.right, right.time, right.mult, right.right);
                }
            } else {
                // Already well-balanced.
                this.left = left;
                this.time = time;
                this.mult = mult;
                this.right = right;
            }


            // This might be optimized and included in the
            // individual cases.
            height = Math.max(left.height, right.height) + 1;
            size = left.size + right.size + mult;
            uniqueSize = left.uniqueSize + right.uniqueSize + 1;
        }
    }

    /**
     * Returns whether the TimeSet is empty.
     *
     * @return whether the TimeSet is empty
     */
    public boolean isEmpty() {
        return mult == 0;
    }

    // The factory method exposes less of the internal structure 
    // of the time set.
    /**
     * Make a new {@code TimeSet} with the given time and multiplicity.
     *
     * @param time the time value
     * @param mult how often that time value occurs in the set
     * @return a newly created {@code TimeSet} initialized with the given time and multiplicity
     */
    public static TimeSet make(double time, int mult) {
        return new TimeSet(EMPTY, time, mult, EMPTY);
    }

    /**
     * Returns the TimeSet's size.
     *
     * @return the TimeSet's size
     */
    public int getSize() {
        return size;
    }

    private TimeSet findNode(double searchTime) {
        TimeSet current = this;
        while (current.mult != 0) {
            if (current.time == searchTime) {
                return current;
            } else if (current.time < searchTime) {
                current = current.right;
            } else {
                current = current.left;
            }
        }

        return EMPTY;
    }

    /**
     * Finds how often a given point in time is included in the TimeSet (its multiplicity).
     *
     * @param searchTime the point in time whose multiplicity should be returned
     * @return how often the given time is included in the TimeSet
     */
    public int multiplicity(double searchTime) {
        return findNode(searchTime).mult;
    }

    /**
     * Returns a copy of this {@code TimeSet} that includes the given {@code newTime} one
     * additional time.
     *
     * @param newTime the point in time to insert into the copy
     * @return a copy of this {@code TimeSet} that includes {@code newTime} one additional time
     */
    public TimeSet including(double newTime) {
        return including(newTime, 1);
    }

    /**
     * Returns a copy of this {@code TimeSet} that includes the given {@code newTime} an additional
     * {@code n} times.
     *
     * @param newTime the point in time to insert into the copy
     * @param n the number of extra times that {@code newTime} should be included in the returned set
     * @return a copy of this TimeSet that includes {@code newTime} an additional {@code n} times
     */
    public TimeSet including(double newTime, int n) {
        if (mult == 0) {
            // This is an empty time set.
            return make(newTime, n);
        }
        if (newTime == time) {
            return new TimeSet(left, time, mult + n, right);
        }

        final TimeSet newLeft;
        final TimeSet newRight;

        // One of the subtrees must be recreated.
        if (newTime < time) {
            newLeft = left.including(newTime, n);
            newRight = right;
        } else {
            newLeft = left;
            newRight = right.including(newTime, n);
        }
        return new TimeSet(newLeft, time, mult, newRight);
    }

    /**
     * Returns a copy of this {@code TimeSet} from which one occurrence of the given {@code oldTime}
     * has been removed.
     *
     * @param oldTime the time to remove once from the copy
     * @return a copy of this {@code TimeSet} from which one occurrence of {@code oldTime} has been removed
     * @throws NoSuchElementException if {@code oldTime} is not contained in the TimeSet
     */
    public TimeSet excluding(double oldTime) {
        if (mult == 0) {
            throw new NoSuchElementException();
        }

        TimeSet newLeft;
        TimeSet newRight;
        double newTime;
        int newMult;

        if (oldTime == time) {
            if (mult > 1) {
                newLeft = left;
                newRight = right;
                newTime = time;
                newMult = mult - 1;
            } else if (right.mult == 0) {
                return left;
            } else if (left.mult == 0) {
                return right;
            } else {
                TimeSetResult reordered = right.extractLeftmost();
                newLeft = left;
                newRight = reordered.tree;
                newTime = reordered.time;
                newMult = reordered.mult;
            }
        } else {
            if (oldTime < time) {
                newLeft = left.excluding(oldTime);
                newRight = right;
            } else {
                newLeft = left;
                newRight = right.excluding(oldTime);
            }
            newTime = time;
            newMult = mult;
        }

        return new TimeSet(newLeft, newTime, newMult, newRight);
    }

    // Must not be called on the empty set.
    private TimeSetResult extractLeftmost() {
        if (mult == 0) {
            throw new RuntimeException("Illegal invocation of extractLeftmost().");
        } else if (left.mult == 0) {
            return new TimeSetResult(right, time, mult);
        } else {
            TimeSetResult lifted = left.extractLeftmost();
            return new TimeSetResult(
                new TimeSet(lifted.tree, time, mult, right), lifted.time, lifted.mult);
        }
    }

    /**
     * Fetches the earliest point in time contained in this collection.
     *
     * @return the earliest point in time contained in this collection
     */
    public double earliestTime() {
        TimeSet tree = this;
        do {
            if (tree.left.mult == 0) {
                return tree.time;
            }
            tree = tree.left;
        } while (true);
    }

    /**
     * Fetches the latest point in time contained in this collection.
     *
     * @return the latest point in time contained in this collection
     */
    public double latestTime() throws NoSuchElementException {
        TimeSet tree = this;
        if (mult == 0) {
            throw new NoSuchElementException();
        }

        do {
            if (tree.right.mult == 0) {
                return tree.time;
            }
            tree = tree.right;
        } while (true);
    }

    /**
     * Fetches the last point in time that allows a certain delay
     * without missing a given deadline. It is not the same to
     * use deadline-delay as the new deadline, as round-off
     * errors might bias the possible values.
     *
     * @param delay the amount by which the result should be able to be delayed without missing the deadline
     * @param deadline the deadline after which checking for a matching point in time will stop
     * @throws NoSuchElementException if the deadline is missed by all points in time with the given delay
     * @return the last point in time that allows the delay without missing the deadline
    */
    public double latestWithDelay(double delay, double deadline) {
        TimeSet current = this;
        TimeSet best = null;
        while (current.mult != 0) {
            if (current.time + delay <= deadline) {
                // The current node is a possible candidate
                // because it can meet the deadline.
                best = current;


                // Proceed down the right subtree to find better estimates.
                // The left subtree has only worse solutions.
                current = current.right;
            } else {
                // Only the left subtree might have solutions.
                current = current.left;
            }
        }

        if (best != null) {
            return best.time;
        } else {
            throw new NoSuchElementException();
        }
    }

    /**
     * Returns the unique contents of this set as an array.
     *
     * @return the unique contents of this set as an array
     */
    public double[] asUniqueArray() {
        double[] result = new double[uniqueSize];
        fillIn(result, 0, true);
        return result;
    }

    /**
     * Returns the contents of this set as a sorted array.
     *
     * @return the contents of this set as a sorted array
     */
    public double[] asArray() {
        double[] result = new double[size];
        fillIn(result, 0, false);
        return result;
    }

    // Copy the contents of this set into an array.
    private void fillIn(double[] result, int start, boolean unique) {
        if (mult == 0) {
            return;
        }
        left.fillIn(result, start, unique);
        if (unique) {
            start = start + left.uniqueSize;
            result[start++] = time;
        } else {
            start = start + left.size;
            for (int i = 0; i < mult; i++) {
                result[start++] = time;
            }
        }
        right.fillIn(result, start, unique);
    }

    /**
     * Computes the earliest time in the set if its earliest n times are each delayed by the given amounts.
     * Requires O(min(size,that.size)).
     *
     * @param that the set of times representing the delays that will be applied to the earliest n points in time,
     *             with n being the size of this given set. They will be applied in order (the smallest delay will be
     *             applied to the highest value etc.).
     * @return the point in time from the earliest n times that is the latest after applying the delay
     */
    public double computeEarliestTime(TimeSet that) {
        return computeEarliestTime(that.asArray());
    }

    /**
     * Computes the earliest time in the set if its earliest n times are each delayed by the given amounts.
     * Requires O(min(size,times.length)).
     *
     * @param delays the values by which the earliest n times are delayed, with n being the length of the array
     * @return the point in time from the earliest n times that is the latest after applying the delay
     */
    public double computeEarliestTime(double[] delays) {
        if (delays.length > size) {
            // The other set contains more elements than I do.
            // I cannot match it in number, no matter how long 
            // we wait.
            return Double.POSITIVE_INFINITY;
        }
        return computeEarliestTime(delays, delays.length - 1);
    }

    // Check part of the necessary comparisons.
    private double computeEarliestTime(double[] delays, int offset) {
        double earliest = Double.NEGATIVE_INFINITY;
        if (mult > 0) {
            earliest = left.computeEarliestTime(delays, offset);
            offset -= left.size;
            if (offset >= 0) {
                // Regardless of the multiplier, we only need to check against
                // the last entry in the array, because the array is ordered.
                earliest = Math.max(earliest, time + delays[offset]);
                offset -= mult;
                if (offset >= 0) {
                    earliest = Math.max(earliest, right.computeEarliestTime(delays, offset));
                }
            }
        }
        return earliest;
    }

    @Override
    public String toString() {
        StringBuffer buffer = new StringBuffer();
        buffer.append("TimeSet(");
        toString(buffer);
        buffer.append(')');
        return buffer.toString();
    }

    private void toString(StringBuffer buffer) {
        if (mult == 0) {
            return;
        } else {
            left.toString(buffer);
            for (int i = 0; i < mult; i++) {
                buffer.append(' ');
                buffer.append(time);
            }
            right.toString(buffer);
        }
    }
}


class TimeSetResult {
    final TimeSet tree;
    final double time;
    final int mult;

    TimeSetResult(TimeSet tree, double time, int mult) {
        this.tree = tree;
        this.time = time;
        this.mult = mult;
    }
}