package de.renew.net;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;


/**
 * A TestTokenBag contains all the tokens contained in a place that are currently tested by transitions.
 */
public class TestTokenBag implements Serializable {
    // I prepare an array with small integers to avoid
    // creating excessive amounts of temporary integer objects.
    static final int maxStaticMult = 15;
    static final Integer[] multiplicities;

    static {
        multiplicities = new Integer[maxStaticMult + 1];
        for (int i = 0; i <= maxStaticMult; i++) {
            multiplicities[i] = Integer.valueOf(i);
        }
    }

    /**
     * The amount of tokens that the TestTokenBag contains.
     */
    private int size;
    /**
     * A Map associating tokens in the TestTokenBag with how often the TestTokenBag contains each of them.
     */
    private Map<Object, Integer> testCount;
    /**
     * A Map associating tokens in the TestTokenBag with the time at which they were initially added to it.
     */
    private Map<Object, Double> orgTime;

    TestTokenBag() {
        testCount = new HashMap<Object, Integer>();
        orgTime = new HashMap<Object, Double>();
        size = 0;
    }

    // Synchronize just to be safe. Maybe this is overly cautious.
    /**
     * Returns the amount of unique tokens in the TestTokenBag.
     *
     * @return the amount of unique tokens in the TestTokenBag
     */
    public synchronized int getUniqueSize() {
        return testCount.size();
    }

    /**
     * Checks how often a given token is included in the TestTokenBag.
     *
     * @param elem the token to check for
     * @return how often {@code elem} is included in the TestTokenBag
     */
    public synchronized int getTestMultiplicity(Object elem) {
        if (testCount.containsKey(elem)) {
            return (testCount.get(elem)).intValue();
        } else {
            return 0;
        }
    }

    private final static Integer toInteger(int result) {
        if (result <= maxStaticMult) {
            return multiplicities[result];
        } else {
            return Integer.valueOf(result);
        }
    }

    /**
     * Gets the unique elements the TestTokenBag contains.
     *
     * @return a Collection containing every unique element of the TestTokenBag
     */
    public synchronized Collection<Object> uniqueElements() {
        // Unfortunately, we have to make a copy of the
        // elements, because the hashtable might be updated
        // asynchronously.
        return new ArrayList<Object>(testCount.keySet());
    }

    /**
     * Checks if a given token is included in the TestTokenBag.
     *
     * @param elem the token to test for
     * @return whether {@code elem} is included in the TestTokenBag
     */
    public synchronized boolean includesTested(Object elem) {
        return testCount.containsKey(elem);
    }

    // A token bag should only be updated under control of its place.
    // Therefore this method is not made public.
    //
    // If this is supposed to change, a place instance may not
    // expose its token bag any longer.
    synchronized void addTested(Object elem, double time) {
        if (testCount.containsKey(elem)) {
            Integer num = testCount.get(elem);
            testCount.put(elem, toInteger(num.intValue() + 1));
        } else {
            // We are testing the first token.
            // For this token, we record its time.
            testCount.put(elem, toInteger(1));
            orgTime.put(elem, Double.valueOf(time));
        }
        ++size;
    }

    // A token bag should only be updated under control of its place.
    // Therefore this method is not made public.
    //
    // If this is supposed to change, a place instance may not
    // expose its token bag any longer.
    synchronized double removeTested(Object elem) {
        if (testCount.containsKey(elem)) {
            int newMult = (testCount.get(elem)).intValue() - 1;
            double time = (orgTime.get(elem)).doubleValue();
            if (newMult == 0) {
                testCount.remove(elem);
                orgTime.remove(elem);
            } else {
                testCount.put(elem, toInteger(newMult));
            }
            --size;
            return time;
        } else {
            throw new RuntimeException("Negative number of tokens detected.");
        }
    }

    @Override
    public String toString() {
        StringBuffer buffer = new StringBuffer();
        buffer.append("TestTokenBag(size: ");
        buffer.append(size);
        buffer.append("; count'token@time:");
        Iterator<Object> enumeration = testCount.keySet().iterator();
        while (enumeration.hasNext()) {
            Object token = enumeration.next();
            buffer.append(' ');
            buffer.append(testCount.get(token));
            buffer.append('\'');
            buffer.append(token);
            buffer.append('@');
            buffer.append(orgTime.get(token));
        }
        buffer.append(')');
        return buffer.toString();
    }
}