package de.renew.net;

import java.io.Serializable;

import de.renew.engine.common.StepIdentifier;
import de.renew.engine.searcher.ChannelTarget;

/**
 * A NetInstance is an instance of a net. It contains instances of net elements.
 */
public interface NetInstance extends ChannelTarget, Serializable {
    /**
     * Confirms the creation of the net instance for the purposes of database storage. This can still be rolled back,
     * but has to be done quite early so that the net is known to the database transaction.
     * Also inserts the initial tokens for all places in the net instance that should receive them early.
     */
    public void earlyConfirmation();

    /**
     * Traces the initial tokens of every place, if they are inserted early on, and adds the {@code NetInstance}
     * to the {@link NetInstanceList}.
     *
     * @param stepIdentifier the identifier of the current simulation step
     */
    public void earlyConfirmationTrace(StepIdentifier stepIdentifier);

    /**
     * Inserts the initial tokens for all places in the net instance that didn't receive them early.
     * Also adds all spontaneous transitions into the {@link de.renew.engine.searchqueue.SearchQueue search queue for
     * possible activated transitions}, allowing a searcher to check them for enabledness.
     *
     * @param stepIdentifier the identifier of the current simulation step
     */
    public void lateConfirmation(StepIdentifier stepIdentifier);

    /**
     * Call this method if the creation of a net instance can be confirmed
     * in one single step.
     *
     * @param stepIdentifier the identifier of the current simulation step
     */
    public void createConfirmation(StepIdentifier stepIdentifier);

    /**
     * Returns the instance of a net element in this NetInstance.
     *
     * @param netObject the net element whose instance should be returned
     * @return the instance of a net element in this NetInstance
     */
    public Object getInstance(Object netObject);

    /**
     * Returns the instance of a {@code Place} in this NetInstance.
     *
     * @param place the {@code Place} whose instance should be returned
     * @return the instance of a {@code Place} in this NetInstance
     */
    public PlaceInstance getInstance(Place place);

    /**
     * Returns the instance of a {@code Transition} in this NetInstance.
     *
     * @param transition the {@code Transition} whose instance should be returned
     * @return the instance of a {@code Transition} in this NetInstance
     */
    public TransitionInstance getInstance(Transition transition);

    /**
     * Returns the Net this is an instance of.
     *
     * @return the Net this is an instance of
     */
    public Net getNet();

    /**
     * Query the ID of the net instance.
     *
     * @return the current ID string
     */
    public String getID();

    /**
     * Set the ID of the net instance. This should only be done
     * during the setup of the net instance, typically after restoring
     * the net from a saved state.
     *
     * @param id the new ID string
     */
    public void setID(String id);

    /**
     * Get the {@code IDRegistry} that is responsible for this net instance.
     *
     * @return the {@code IDRegistry} that is responsible for this net instance
     */
    public IDRegistry getRegistry();
}