package de.renew.engine.searcher;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import de.renew.engine.simulator.SimulationThreadPool;


/**
 * Primary implementation of {@link TriggerCollection}.
 */
public class TriggerCollectionImpl implements Serializable, TriggerCollection {
    // Keep track of triggering objects.
    private final Set<TriggerableCollection> triggers;
    private final Triggerable triggerable;

    public TriggerCollectionImpl(Triggerable triggerable) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        triggers = new HashSet<>();
        this.triggerable = triggerable;
    }

    // The following two method must only be accessed through
    // the include and exclude methods of triggerable collections,
    // otherwise deadlocks might occur. Those method will make sure
    // to lock this object.
    @Override
    public void include(TriggerableCollection trigger) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        triggers.add(trigger);
    }

    @Override
    public void exclude(TriggerableCollection trigger) {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        triggers.remove(trigger);
    }

    @Override
    public void clear() {
        assert SimulationThreadPool.isSimulationThread() : "is not in a simulation thread";
        List<TriggerableCollection> list;
        synchronized (this) {
            // Get the triggers.
            list = new ArrayList<>(triggers);
        }


        // Send the notifications. No need to synchronize here.
        // Better allow concurrent accesses to reduce deadlock potential.
        // The triggerable collections are required to do the locking.
        for (TriggerableCollection triggerables : list) {
            triggerables.exclude(triggerable);
        }
    }
}