package de.renew.application;


import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Properties;

import de.renew.database.SetupHelper;
import de.renew.engine.simulator.SimulationThreadPool;
import de.renew.engine.simulator.Simulator;
import de.renew.net.NetInstance;
import de.renew.net.NetNotFoundException;
import de.renew.net.loading.Finder;
import de.renew.net.loading.NetLoader;
import de.renew.net.loading.PathlessFinder;
import de.renew.shadow.DefaultCompiledNetLoader;
import de.renew.shadow.DefaultShadowNetLoader;
import de.renew.shadow.ShadowLookup;
import de.renew.shadow.ShadowNetSystem;
import de.renew.shadow.SyntaxException;
import de.renew.util.SingletonException;

/**
 * This interface is part of the Simulator's interface and may be used by other Plugins.
 * All implementations must allow the following functionality:
 * <ul>
 *     <li>Addition and removal of {@link SimulatorExtension}</li>
 *     <li>Setup and termination of a simulation</li>
 *     <li>Retrieval of information regarding the current simulation mode</li>
 *     <li>The insertion of a {@link de.renew.shadow.ShadowNetSystem} that should be simulated</li>
 *     <li>The loading and persisting of the current simulation state to a database</li>
 *     <li>Setting the  {@link de.renew.net.loading.NetLoader} that is need to load nets</li>
 *     <li>Setting and removing the {@link de.renew.net.loading.Finder} that is needed to find nets</li>
 * </ul>
 */
public interface SimulatorManager {

    /**
     * The name of the property to get the combined simulator mode from,
     * determining simulator class and multiplicity.
     */
    String MODE_PROP_NAME = "de.renew.simulatorMode";

    /**
     * The name of the property to get the priority of the simulator thread
     */
    String PRIORITY_PROP_NAME = "de.renew.simulatorPriority";

    /**
     * The name of the property to get the simulator multiplicity from.
     */
    String MULTIPLICITY_PROP_NAME = "de.renew.simulatorMultiplicity";

    /**
     * The name of the property to get the simulator class name from.
     */
    String CLASS_PROP_NAME = "de.renew.simulatorClass";

    /**
     * The name of the property to get the eager simulation flag from.
     */
    String EAGER_PROP_NAME = "de.renew.eagerSimulation";

    /**
     * The name of the property to get the flag from that tells whether custom
     * classes should be reloaded every simulation run.
     */
    String REINIT_PROP_NAME = "de.renew.classReinit";

    /**
     * Registers an extension to the simulation component. The extension will
     * become active the next time a new simulation environment is set up.
     * Active extensions will be notified about several events (see
     * {@link SimulatorExtension} interface).
     *
     * @param extension the extension to register. Duplicate registrations will be
     *        ignored.
     * @throws SingletonException if this object is not the simulator plugin singleton
     *         instance anymore.
     */
    void addExtension(SimulatorExtension extension);

    /**
     * Removes and deregisters the given {@code SimulatorExtension} from the simulation engine.
     * The deregistration does not cancel active extensions from any currently running simulation,
     * but they will not be activated again when the next simulation is set up.
     *
     * @param extension the extension to deregister. If the extension was not
     *        registered before, this method call is ignored.
     * @throws SingletonException if this object is not  the simulator plugin singleton
     *         instance anymore.
     */
    void removeExtension(SimulatorExtension extension);

    /**
     * Sets up a new simulation environment by doing the following:
     * <ul>
     * <li>the simulation thread pool is reset</li>
     * <li>the set of properties is configured</li>
     * <li>all registered extensions become activated</li>
     * <li>a new simulation engine is set up</li>
     * <li>the initial net instance is created</li>
     * </ul>
     * The simulation will <b>not</b> be started, so no steps will be executed.
     * All added {@link SimulatorExtension}s will be notified.
     *
     * <p>
     * This method will automatically create a new thread if it is not called
     * from a simulation thread. The disadvantage is that exceptions are not
     * communicated.
     * </p>
     *
     * <p>
     * The behavior of the method has changed from Renew release 2.1 to 2.2. It
     * no longer automatically terminates a running simulation. Instead, an
     * exception is thrown (see below).
     * </p>
     *
     * <p>
     * Callers of this method that switch to a simulation thread by themselves
     * should use {@link SimulationThreadPool#getNew()}. This method
     * automatically discards the new thread pool if simulation setup fails.
     * After successful execution of this method, the new thread pool becomes
     * the current thread pool and the calling thread belongs to the current
     * simulation.
     * </p>
     *
     * <p>
     * Access to this method is exclusive. The Java synchronized mechanism is
     * replaced by a specialized {@link de.renew.util.Lock#lock()}. How to achieve
     * synchronization across multiple methods is explained there.
     * </p>
     *
     * @param props additional properties that specify this simulation
     *        environment. These properties will override any default values
     *        from the plugin properties. May be <code>null</code>.
     * @throws SingletonException if this object is not the simulator plugin
     *         singleton instance anymore.
     * @throws SimulationRunningException if a simulation is already running.
     */
    void setupSimulation(final Properties props);

    /**
     * Terminates the current simulation. If no simulation has been set up,
     * nothing happens.
     *
     * <p>
     * Access to this method is exclusive. The Java synchronized mechanism is
     * replaced by a specialized {@link de.renew.util.Lock#lock()}. How to achieve
     * synchronization across multiple methods is explained there.
     * </p>
     *
     * @throws SingletonException if this object is not the simulator plugin singleton
     *         instance anymore.
     * <p>
     */
    void terminateSimulation();

    /**
     * Returns whether the simulation is currently active. If this method returns
     * {@code true}, a simulation has been set up <i>and</i> is still
     * active (see documentation of {@link Simulator#isActive}).
     * <p>
     * This method will automatically create a new thread if it is not called
     * from a simulation thread
     *
     * @return {@code true}, if the simulation is active (see above), {@code false} otherwise.
     * @throws SingletonException if this object is not the simulator plugin singleton
     *         instance anymore.
     */
    boolean isSimulationActive();

    /**
     * Returns the current simulation environment, if a simulation has been set
     * up.
     * <p>
     * Do not expect that the data provided here has any guaranteed life span -
     * if you want to be informed about the termination of the simulation, write
     * and register a {@link SimulatorExtension}.
     * </p>
     *
     * @return a <code>SimulationEnvironment</code> object describing the actual
     *         simulation setup.
     * @throws SingletonException if this object is not the simulator plugin singleton
     *         instance anymore.
     */
    SimulationEnvironment getCurrentEnvironment();

    /**
     * Adds net templates based on the shadow nets in the given {@code ShadowNetSystem} to the current
     * simulation.
     * <p>
     * When this method is called on a fresh simulation setup, the given net
     * system's information about shadow net loader and compiler are extracted
     * and kept for the simulation lifetime. In this case, the given {@code ShadowNetSystem}
     * <i>must</i> be configured with a {@code ShadowNetCompiler}. However,
     * setting a {@code ShadowNetLoader} is optional.
     * </p>
     *
     * <p>
     * Access to this method is exclusive. The Java synchronized mechanism is
     * replaced by a specialized {@link de.renew.util.Lock#lock()}. How to achieve
     * synchronization across multiple methods is explained there.
     * </p>
     *
     * @param netSystem holds all nets to be compiled into this simulation
     *        environment. The state of the net system will change during
     *        the insertion process: nets are marked as compiled, and the
     *        default shadow net loader is configured (optional).
     * @return the lookup resulting from the compilation.
     * @throws SyntaxException if an error occurs during the compilation process.
     * @throws NullPointerException if the {@code netSystem} is {@code null}.
     * <p>
     * This exception could also be thrown if the
     * {@link DefaultCompiledNetLoader} has been requested for
     * this simulation setup (via {@link #setDefaultNetLoader}),
     * this is the first {@code netSystem} and there is no
     * net loader configured within the net system.
     * </p>
     * @throws NoSimulationException if there is no simulation set up.
     * @throws SingletonException if this object is not the simulator plugin
     *         singleton instance anymore.
     */
    ShadowLookup insertNets(final ShadowNetSystem netSystem)
        throws SyntaxException, NoSimulationException;

    /**
     * Creates a net instance within the current simulation.
     *
     * <p>
     * Access to this method is exclusive. The Java synchronized mechanism is
     * replaced by a specialized {@link de.renew.util.Lock#lock()}. How to achieve
     * synchronization across multiple methods is explained there.
     * </p>
     *
     * @param net the name of the net template for the net instance to build. If
     *        {@code null}, then no net instance will be created.
     * @return the created netInstance. Returns {@code null} if the
     *         instance creation failed.
     * @throws NetNotFoundException if the instance creation failed because no net with the given
     *         name could be found.
     * @throws NoSimulationException if there is no simulation set up.
     * @throws SingletonException if this object is not the simulator plugin singleton
     *         instance anymore.
     */
    NetInstance createNetInstance(final String net)
        throws NetNotFoundException, NoSimulationException;

    /**
     * If the corresponding properties have been set, the simulation will be
     * connected to a database engine and the last state from the database will
     * be restored. The net structures on which the state is based have to be
     * known before this method is called.
     * <p>
     * If the database properties are <i>not</i> set, this method will return
     * silently without modifying the simulation state. The return value will
     * then be the
     * {@link de.renew.database.SetupHelper.SimulationState#TERMINATED_STATE}
     * object. This object is also returned when the database connection failed.
     * If the database is configured correctly, the result will be an individual
     * state object, whether or not there was a state to restore.
     * </p>
     * <p>
     * If this method is called after the current simulation has already
     * performed some steps, i.e. created its own state, the result is
     * undefined.
     * </p>
     *
     * <p>
     * Access to this method is exclusive. The Java synchronized mechanism is
     * replaced by a specialized {@link de.renew.util.Lock#lock()}. How to achieve
     * synchronization across multiple methods is explained there.
     * </p>
     *
     * @return A {@code SimulationState} object describing the result of
     *         the operation (see comments above).
     * @throws NoSimulationException if there is no simulation set up.
     * @throws SingletonException if this object is not the simulator plugin singleton
     *         instance anymore.
     * @see SetupHelper#setup
     */
    SetupHelper.SimulationState restoreStateFromDatabase();

    /**
     * Writes the given {@code NetInstance}s as well as all known
     * {@code Net}s and the {@code SearchQueue} contents and some
     * additional information to the stream. The written information is
     * sufficient to continue the simulation from the same state after
     * deserialization.
     * <p>
     * If the given {@code ObjectOutput} is a <b>
     * {@link de.renew.util.RenewObjectOutputStream}</b>, its feature of cutting
     * down the recursion depth by delaying the serialization of some fields
     * will be used.
     * </p>
     *
     * <p>
     * Access to this method is exclusive. The Java synchronized mechanism is
     * replaced by a specialized {@link de.renew.util.Lock#lock()}. How to achieve
     * synchronization across multiple methods is explained there.
     * </p>
     *
     * @param output target stream (see note about
     *        {@code RenewObjectOutputStream} above).
     * @param instances an array of net instances to be explicitly included in the
     *        saved state (e.g. instances displayed to the user).
     * @throws IOException if an error occurs during the serialisation to the output
     *         stream.
     * @throws NoSimulationException if there is no simulation set up.
     * @throws SingletonException if this object is not the simulator plugin singleton
     *         instance anymore.
     */
    void saveState(final ObjectOutput output, final NetInstance[] instances) throws IOException;

    /**
     * Restores a simulation saved by {@link #saveState}. Reads all stored
     * {@code Net}s and {@code NetInstance}s. The result is a
     * ready-to-run simulation setup. Some net instances (those that have been
     * explicitly referenced in the {@code saveState} call) are returned so
     * they can be processed specially (e.g. open instance drawing windows).
     * <p>
     * If the given {@code ObjectInput} is a <b>
     * {@link de.renew.util.RenewObjectInputStream}</b>, the necessary steps to
     * cover delayed serialization will be made. The ObjectInputStream will be
     * read using {@code de.renew.util.ClassSource} to provide its ability
     * of reloading all user defined classes.
     * </p>
     *
     * <p>
     * This method will automatically create a new thread if it is not called
     * from a simulation thread. Contrary to the method
     * {@link #setupSimulation(Properties)}, exceptions are communicated anyway.
     * </p>
     *
     * <p>
     * The behavior of the method has changed from Renew release 2.1 to 2.2. It
     * no longer automatically terminates a running simulation. Instead, an
     * exception is thrown (see below).
     * </p>
     *
     * <p>
     * Callers of this method that switch to a simulation thread by themselves
     * should use {@link SimulationThreadPool#getNew()}. This method
     * automatically discards the new thread pool if simulation setup fails.
     * After successful execution of this method, the new thread pool becomes
     * the current thread pool and the calling thread belongs to the current
     * simulation.
     * </p>
     *
     * <p>
     * Access to this method is exclusive. The Java synchronized mechanism is
     * replaced by a specialized {@link de.renew.util.Lock#lock()}. How to achieve
     * synchronization across multiple methods is explained there.
     * </p>
     *
     * @param input source stream (see note about
     *        {@code RenewObjectInputStream} above).
     * @param props additional properties that specify this simulation
     *        environment. These properties will override any default values
     *        from the plugin properties. May be {@code null}.
     * @return array of explicitly stored net instances (e.g. to be displayed to
     *         the user).
     * @throws IOException if an error occurs during reading the input stream
     * @throws ClassNotFoundException if an unknown object type was included in the state
     * @throws SingletonException if this object is not the simulator plugin
     *         singleton instance anymore.
     * @throws SimulationRunningException if a simulation is already running.
     */
    NetInstance[] loadState(final ObjectInput input, final Properties props)
        throws IOException, ClassNotFoundException, SimulationRunningException;

    // Setting the  {@link de.renew.net.loading.NetLoader} that is need to load nets

    /**
     * Configure the {@code NetLoader} to use in the next simulation setup.
     * There is no way to change the net loader in an already set up simulation.
     *
     * @param loader the {@code NetLoader} to use in the next simulation. If
     *        {@code null}, no net loader will be used during the next
     *        simulation.
     * @throws SingletonException if this object is not the simulator plugin singleton
     *         instance anymore.
     */
    void setNetLoader(NetLoader loader);

    /**
     * Make sure that the user-level classes get reloaded for the next compiler
     * run, if the class reinit mode has been configured. This method is called
     * automatically by {@link #setupSimulation}.
     * <p>
     * This method will automatically create a new thread if it is not called
     * from a simulation thread
     *
     * <p>
     * Access to this method is exclusive. The Java synchronized mechanism is
     * replaced by a specialized {@link de.renew.util.Lock#lock()}. How to achieve
     * synchronization across multiple methods is explained there.
     * </p>
     *
     * @param props the configuration to extract the {@link #REINIT_PROP_NAME}
     *        property from.
     * @throws IllegalStateException if simulation is already running
     * @throws IllegalStateException if there is an active simulation.
     * @throws SingletonException if this object is not the simulator plugin
     *         singleton instance anymore.
     */
    void possiblySetupClassSource(final Properties props) throws IllegalStateException;

    /**
     * Configure the {@link DefaultCompiledNetLoader} to be used in the next
     * simulation setup. This net loader will be connected to the plugin's
     * shadow net system.
     *
     * @throws SingletonException if this object is not the simulator plugin singleton
     *         instance anymore.
     */
    void setDefaultNetLoader();

    /**
     * Registers a finder for the default shadow net loader. Finders are without
     * effect if the default shadow net loader is overridden (either as plug-in
     * configuration or by an individual shadow net system).
     *
     * @param finder the shadow net file finder to add to the default shadow net
     *        loader.
     * @see DefaultShadowNetLoader#registerFinder
     */
    void registerDefaultNetFinder(final Finder finder);

    /**
     * Deregisters a finder from the default shadow net loader.
     *
     * @param finder the shadow net file findet to remove from the default shadow
     *        net loader.
     * @see DefaultShadowNetLoader#removeFinder
     */
    void removeDefaultNetFinder(final Finder finder);

    /**
     * Registers a pathless finder for the default shadow net loader. Finders
     * are without effect if the default shadow net loader is overridden (either
     * as plug-in configuration or by an individual shadow net system).
     *
     * @param finder the shadow net file finder to add to the default shadow net
     *        loader.
     * @see DefaultShadowNetLoader#registerPathlessFinder
     */
    void registerDefaultPathlessFinder(final PathlessFinder finder);

    /**
     * Deregisters a pathless finder from the default shadow net loader.
     *
     * @param finder the shadow net file finder to remove from the default shadow
     *        net loader.
     * @see DefaultShadowNetLoader#removePathlessFinder
     */
    void removeDefaultPathlessFinder(final PathlessFinder finder);
}
