package de.renew.gui.tool;

import java.awt.Rectangle;
import java.awt.event.MouseEvent;

import CH.ifa.draw.framework.DrawingEditor;
import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.framework.FigureEnumeration;
import CH.ifa.draw.standard.CreationTool;
import de.renew.gui.CPNTextFigure;
import de.renew.gui.TransitionFigure;
import de.renew.gui.VirtualTransitionFigure;


/**
 * Instances of this class create new {@link VirtualTransitionFigure}s
 * and are called upon by the user through the tool buttons within the palette.
 */
public class VirtualTransitionFigureCreationTool extends CreationTool {

    /**
     * the currently created figure
     */
    private VirtualTransitionFigure virtualTransition = null;

    /**
     * The constructor for the creation tool.
     * @param editor The editor this tool in used in.
     */
    public VirtualTransitionFigureCreationTool(DrawingEditor editor) {
        super(editor);
    }

    /**
     * Creates a new VirtualTransitionFigure.
     */
    @Override
    protected Figure createFigure() {
        return new VirtualTransitionFigure(null);
    }

    /**
     * Handles mouse move events in the drawing view.
     */
    @Override
    public void mouseMove(MouseEvent e, int x, int y) {}

    /**
     * If the mouse down hits a TransitionFigure create a new VirtualTransitionFigure.
     */
    @Override
    public void mouseDown(MouseEvent e, int x, int y) {
        Figure fTarget = drawing().findFigure(x, y);
        if (fTarget instanceof TransitionFigure semanticTransitionFigure) {

            // take the original Transition as semantic Transition figure if fTarget is a VirtualTransitionFigure
            while (semanticTransitionFigure instanceof VirtualTransitionFigure virtualTransitionFigure) {
                semanticTransitionFigure = virtualTransitionFigure.getSemanticFigure();
            }

            virtualTransition = new VirtualTransitionFigure(semanticTransitionFigure);
            view().add(virtualTransition);

            FigureEnumeration children = ((TransitionFigure) fTarget).children();
            Rectangle r = fTarget.displayBox();

            // clone all name inscription from fTarget which may be a VirtualTransitionFigure
            while (children.hasMoreElements()) {
                Figure nextFigure = children.nextFigure();
                if (nextFigure instanceof CPNTextFigure textFig
                    && textFig.getType() == CPNTextFigure.NAME) {

                    CPNTextFigure virtualTransitionName = new CPNTextFigure(CPNTextFigure.NAME);
                    virtualTransitionName.setText(textFig.getText());
                    view().add(virtualTransitionName);
                    virtualTransitionName.setParent(virtualTransition);
                    // calculate text offset
                    int xDistance = textFig.displayBox().x - r.x;
                    int yDistance = textFig.displayBox().y - r.y;

                    virtualTransitionName.moveBy(xDistance, yDistance);
                }
            }
            virtualTransition.displayBox(new Rectangle(r.x, r.y, r.width, r.height));
            changesMade();
        }
    }

    /**
     * Adjust the created connection or split segment.
     */
    @Override
    public void mouseDrag(MouseEvent e, int x, int y) {
        if (virtualTransition != null) {
            Rectangle r = virtualTransition.displayBox();
            virtualTransition.displayBox(
                new Rectangle(x - (r.width / 2), y - (r.height / 2), r.width, r.height));
        }
    }

    /**
     * Drop new VirtualTransitionFigure.
     */
    @Override
    public void mouseUp(MouseEvent e, int x, int y) {
        virtualTransition = null;
        editor().toolDone();
    }

    @Override
    public void deactivate() {
        super.deactivate();
        virtualTransition = null;
    }
}
