package de.renew.gui;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Enumeration;
import java.util.Hashtable;


public class PatchingStorableInput extends CH.ifa.draw.util.StorableInput {

    /**
     * enables patching when a string attribute is read.
     */
    private final boolean stringReadPatching = true;

    /**
     * maps prefixes which must be replaced by another prefix.
     */
    private static final Hashtable<String, String> patchPrefixes = new Hashtable<>();

    static {
        patchPrefixes.put("CH.ifa.draw.cpn.", "de.renew.gui.");
    }

    /**
     * maps classes which must be replaced by other classes.
     */
    private static final Hashtable<String, String> patchMap = new Hashtable<>();

    static {
        ReplaceHelper toGui = new ReplaceHelper("de.renew.gui");
        toGui.registerAll(
            "de.renew.gui.fs.AssocArrowTip", "de.renew.gui.fs.IsaArrowTip",
            "de.renew.diagram.AssocArrowTip", "de.renew.fa.figures.AssocArrowTip");
    }

    PatchingStorableInput(InputStream stream, boolean useUTF) {
        super(stream, useUTF);
    }

    PatchingStorableInput(File file, boolean useUTF) throws FileNotFoundException {
        super(file, useUTF);
    }

    /**
     * Initializes a Storable input with the given string.
     */
    public PatchingStorableInput(String stringStream) {
        super(stringStream);
    }

    public PatchingStorableInput(URL location, boolean useUTF) throws IOException {
        super(location, useUTF);
    }

    @Override
    protected Object makeInstance(String className) throws IOException {
        return super.makeInstance(patchString(className));
    }

    @Override
    public String readString() throws IOException {
        String s = super.readString();

        if (stringReadPatching) {
            s = patchString(s);
        }

        return s;
    }

    protected String patchString(String s) {
        boolean patched = false;
        Enumeration<String> e = patchPrefixes.keys();
        while (e.hasMoreElements() && !patched) {
            String patchPrefix = e.nextElement();
            if (s.startsWith(patchPrefix)) {
                s = patchPrefixes.get(patchPrefix) + s.substring(patchPrefix.length());
                patched = true;
            }
        }
        String patch = patchMap.get(s);
        if (patch != null) {
            s = patch;
        }

        return s;
    }

    /**
     * Helper class to allow registering several classes to the {@code patchMap} at once.
     * @param newPackage the new package's name, with no trailing dot
     */
    private record ReplaceHelper(String newPackage) {

        /**
         * Adds all fully classified listed classes to the {@code patchMap}, to be patched to {@code newPackage}.
         * @param classNames the fully classified class names to be registered
         */
        public void registerAll(String... classNames) {
            for (String className : classNames) {
                patchMap.put(className, convert(className));
            }
        }

        /**
         * Adds all classes in the given package to the {@code patchMap}, to be patched to {@code newPackage}.
         * @param packageName the old package name, with no trailing dot
         * @param classNames the class names to be registered, without the package name/path
         */
        public void registerAllInPackage(String packageName, String... classNames) {
            if (packageName.endsWith("."))
                packageName = packageName.substring(0, packageName.length() - 1);
            for (String className : classNames) {
                String fullClassName = packageName + "." + className;
                patchMap.put(fullClassName, convert(fullClassName));
            }
        }

        /**
         * Converts a fully classified class name to the new package.
         * @param fullClassName the fully classified class name
         * @return the converted, fully classified class name
         */
        public String convert(String fullClassName) {
            if (newPackage == null || newPackage.isEmpty()) {
                throw new NullPointerException("newPackage is null or empty");
            }
            String className = fullClassName.substring(fullClassName.lastIndexOf('.') + 1);
            return newPackage + "." + className;
        }
    }
}