package CH.ifa.draw.application;

import javax.swing.JMenu;
import javax.swing.JMenuItem;

import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.junit.jupiter.MockitoExtension;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.mockito.Mockito.mock;


@ExtendWith(MockitoExtension.class)
public class MenuManagerTest {

    private static MenuManager testee;

    private String topLevelSingleId;
    private static int TOP_LEVEL_COUNTER = 0;

    @BeforeAll
    public static void setUpAll() {
        testee = MenuManager.getInstance();
        final DrawApplication gui = mock(DrawApplication.class);
        testee.setGui(gui);
    }

    @BeforeEach
    public void setUp() {
        TOP_LEVEL_COUNTER++;
        topLevelSingleId = "parentTestMenu" + TOP_LEVEL_COUNTER;
    }

    @Test
    public void testInitialization() {
        // when
        final MenuManager testee = MenuManager.getInstance();

        // then
        assertNotNull(testee);
    }

    @Test
    public void testMenuItemExistsAfterRegisteringIt() {
        // given
        final JMenuItem testItem = new JMenuItem();
        testItem.putClientProperty(MenuManager.ID_PROPERTY, "testValue");

        // when
        testee.registerMenu("testParent", testItem);

        // then
        final JMenu menu = testee.getToplevelMenu("testParent");
        assertNotNull(menu);
        assertEquals("testParent", menu.getText());
    }

    @Test
    public void testUnregisterMenuButTopLevelRemains() {
        // given
        final String idOfMenu = "menuItem1";
        final JMenuItem testItem1 = new JMenuItem();
        testItem1.putClientProperty(MenuManager.ID_PROPERTY, idOfMenu);
        final JMenuItem testItem2 = new JMenuItem();
        testItem2.putClientProperty(MenuManager.ID_PROPERTY, "menuItem2");
        testee.registerMenu(topLevelSingleId, testItem1);
        testee.registerMenu(topLevelSingleId, testItem2);

        // when
        testee.unregisterMenu(idOfMenu);

        // then
        final JMenu topLevelMenu = testee.getToplevelMenu(topLevelSingleId);
        assertEquals(1, topLevelMenu.getItemCount());
        assertEquals(testItem2, topLevelMenu.getItem(0));
    }

    @Test
    public void testUnregisterMenuRemoveTrailingSeparators() {
        // given
        final String idOfMenu = "menuItem1";
        final JMenuItem testItem1 = new JMenuItem();
        testItem1.putClientProperty(MenuManager.ID_PROPERTY, idOfMenu);
        final JMenuItem testItem2 = new JMenuItem();
        testItem2.putClientProperty(MenuManager.ID_PROPERTY, "menuItem2");
        final JMenuItem seperator = MenuManager.createSeparator("separator1");
        seperator.putClientProperty(MenuManager.ID_PROPERTY, "sep1");

        testee.registerMenu(topLevelSingleId, testItem1);
        testee.registerMenu(topLevelSingleId, testItem2);
        testee.registerMenu(topLevelSingleId, seperator);

        // when
        testee.unregisterMenu(testItem1);

        // then
        final JMenu topLevelMenu = testee.getToplevelMenu(topLevelSingleId);
        assertEquals(1, topLevelMenu.getItemCount());
        assertEquals(testItem2, topLevelMenu.getItem(0));
    }

    @Test
    public void testUnregisterMenuRemoveLeadingSeparators() {
        // given
        final String idOfMenu = "menuItem1";
        final JMenuItem testItem1 = new JMenuItem();
        testItem1.putClientProperty(MenuManager.ID_PROPERTY, idOfMenu);
        final JMenuItem seperator = MenuManager.createSeparator("separator1");
        seperator.putClientProperty(MenuManager.ID_PROPERTY, "sep1");
        final JMenuItem testItem2 = new JMenuItem();
        testItem2.putClientProperty(MenuManager.ID_PROPERTY, "menuItem2");

        testee.registerMenu(topLevelSingleId, testItem1);
        testee.registerMenu(topLevelSingleId, seperator);
        testee.registerMenu(topLevelSingleId, testItem2);

        // when
        testee.unregisterMenu(testItem1);

        // then
        final JMenu topLevelMenu = testee.getToplevelMenu(topLevelSingleId);
        assertEquals(1, topLevelMenu.getItemCount());
        assertEquals(testItem2, topLevelMenu.getItem(0));
    }

    @Test
    public void testRemoveTopLevelMenuIfLastItemIsUnregistered() {
        // given
        final String menuItem1 = "menuItem1";
        final JMenuItem testItem1 = new JMenuItem();
        testItem1.putClientProperty(MenuManager.ID_PROPERTY, menuItem1);

        testee.registerMenu(topLevelSingleId, testItem1);

        // when
        testee.unregisterMenu(menuItem1);

        // then
        final JMenu topLevelMenu = testee.getToplevelMenu(topLevelSingleId);
        assertNull(topLevelMenu);
    }

    @Test
    public void testRegisterMenu() {
        //given
        JMenuItem menuItem1 = new JMenuItem();
        JMenuItem menuItem2 = new JMenuItem();
        final String testItem1 = "menuItem1";
        final String testItem2 = "menuItem2";

        menuItem1.putClientProperty(MenuManager.ID_PROPERTY, testItem1);
        menuItem2.putClientProperty(MenuManager.ID_PROPERTY, testItem2);
        //when
        testee.registerMenu(topLevelSingleId, menuItem1, 1, 2);
        testee.registerMenu(topLevelSingleId, menuItem2, 1, 1);

        //then
        JMenu topLevelMenu = testee.getToplevelMenu(topLevelSingleId);
        assertEquals(menuItem1, topLevelMenu.getItem(0));
        assertEquals(menuItem2, topLevelMenu.getItem(1));
    }

    @Test
    public void testRegisterMenuWhereMiddle() {
        //given
        JMenuItem menuItem1 = new JMenuItem();
        JMenuItem menuItem2 = new JMenuItem();
        JMenuItem menuItem3 = new JMenuItem();
        final String testItem1 = "menuItem1";
        final String testItem2 = "menuItem2";
        final String testItem3 = "menuItem3";

        menuItem1.putClientProperty(MenuManager.ID_PROPERTY, testItem1);
        menuItem2.putClientProperty(MenuManager.ID_PROPERTY, testItem2);
        menuItem3.putClientProperty(MenuManager.ID_PROPERTY, testItem3);

        testee.registerMenu(topLevelSingleId, menuItem1, 1, 0);
        testee.registerMenu(topLevelSingleId, menuItem2, 1, 1);

        //when
        testee.registerMenu(topLevelSingleId, menuItem3, 1, 2);

        //then
        JMenu topLevelMenu = testee.getToplevelMenu(topLevelSingleId);
        assertEquals(menuItem1, topLevelMenu.getItem(0));
        assertEquals(menuItem2, topLevelMenu.getItem(2));
        assertEquals(menuItem3, topLevelMenu.getItem(1));
    }

    @Test
    public void testUnregisterMenuReduceAdjacentSeparatorsToOne() {
        // given
        final JMenuItem testItem1 = new JMenuItem();
        testItem1.putClientProperty(MenuManager.ID_PROPERTY, "menuItem1");
        final JMenuItem testItem2 = new JMenuItem();
        testItem2.putClientProperty(MenuManager.ID_PROPERTY, "menuItem2");

        final JMenuItem separator1 = MenuManager.createSeparator("separator1");
        separator1.putClientProperty(MenuManager.ID_PROPERTY, "sep1");
        final JMenuItem separator2 = MenuManager.createSeparator("separator2");
        separator2.putClientProperty(MenuManager.ID_PROPERTY, "sep2");
        final JMenuItem separator3 = MenuManager.createSeparator("separator3");
        separator3.putClientProperty(MenuManager.ID_PROPERTY, "sep3");

        final JMenuItem testItem3 = new JMenuItem();
        testItem3.putClientProperty(MenuManager.ID_PROPERTY, "menuItem3");


        testee.registerMenu(topLevelSingleId, testItem1);
        testee.registerMenu(topLevelSingleId, testItem2);
        testee.registerMenu(topLevelSingleId, separator1);
        testee.registerMenu(topLevelSingleId, separator2);
        testee.registerMenu(topLevelSingleId, separator3);
        testee.registerMenu(topLevelSingleId, testItem3);


        // when
        testee.unregisterMenu(testItem1);

        // then
        final JMenu topLevelMenu = testee.getToplevelMenu(topLevelSingleId);
        assertEquals(3, topLevelMenu.getItemCount());
        assertEquals(testItem2, topLevelMenu.getItem(0));
    }
}
