/*
 * @(#)StorableInOut.java 5.1
 *
 */
package CH.ifa.draw.util;

import java.net.URI;
import java.util.Vector;


/**
 * An abstract superclass of storable input and output streams.
 * StorableInOut handles the object identity of the stored objects
 * through a map and preserves the original filename.
 * <br>
 * {@link Storable}
 * {@link StorableInput}
 * {@link StorableOutput}
 */
public abstract class StorableInOut {
    private Vector<Storable> fMap = new Vector<Storable>();
    private URI location;

    /**
     * Constructs a StorableInOut to save storables in a Vector as well as
     * the location where the storables are saved.
     */
    protected StorableInOut() {
        this(null);
    }

    /**
     * Constructs a StorableInOut to save storables in a Vector as well as
     * the location where the storables are saved.
     *
     * @param location the URI of all storables
     */
    protected StorableInOut(URI location) {
        this.location = location;
    }

    /**
     * Getter for the URI where all storables are saved.
     *
     * @return the URI of the storable
     */
    public URI getURI() {
        return location;
    }

    /**
     * Indicates whether the storable is saved in this class' Vector.
     *
     * @param storable the storable to check
     * @return <code>true</code>, if the storable is currently stored
     */
    protected boolean mapped(Storable storable) {
        return fMap.contains(storable);
    }

    /**
     * Adds a storable to the Vector. Each storable can only be added once.
     *
     * @param storable the storable that will be added
     */
    protected void map(Storable storable) {
        if (!fMap.contains(storable)) {
            fMap.addElement(storable);
        }
    }

    /**
     * Gets the storable from the Vector.
     *
     * @param ref the index of the storable
     * @return the storable at the provided index
     */
    protected Storable retrieve(int ref) {
        return fMap.elementAt(ref);
    }

    /**
     * Gets the index of the storable in the Vector.
     *
     * @param storable the storable the index will be looked up for
     * @return <code>index</code> of the storable if it was found<br>
     *         <code>-1</code> else
     */
    protected int getRef(Storable storable) {
        return fMap.indexOf(storable);
    }
}