package de.renew.net;

import de.renew.engine.common.StepIdentifier;

import de.renew.net.event.PlaceEventListener;
import de.renew.net.event.PlaceEventListenerSet;
import de.renew.net.event.PlaceEventProducer;

import de.renew.util.Lock;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.util.Set;


/**
 * A representation of a place instance. <p/>
 *
 * This abstract class contains all functionality that is needed to display a place instance with the <code>remote</code> plugin.
 * It abstracts from its implementation, especially from the functionality that is used by the simulation.
 *
 * @author Michael Simon
 */
public abstract class PlaceInstance
                implements PlaceEventProducer, Serializable {
    static final long serialVersionUID = 0L;

    protected NetInstance netInstance;
    protected Place place;

    /**
     * The lock that controls access to this place instance.
     */
    public transient Lock lock = new Lock();

    /**
     * The listeners that want to be informed about every update
     * of the current marking.
     **/
    protected transient PlaceEventListenerSet listeners = new PlaceEventListenerSet();

    PlaceInstance(NetInstance netInstance, Place place) {
        this.netInstance = netInstance;
        this.place = place;
    }

    public NetInstance getNetInstance() {
        return netInstance;
    }

    public Place getPlace() {
        return place;
    }

    @Override
    public void addPlaceEventListener(PlaceEventListener listener) {
        listeners.addPlaceEventListener(listener);
    }

    @Override
    public void removePlaceEventListener(PlaceEventListener listener) {
        listeners.removePlaceEventListener(listener);
    }

    /**
     * Returns the set of currently untested tokens.
     */
    public abstract Set<Object> getDistinctTokens();

    public abstract Set<Object> getDistinctTestableTokens();

    public abstract int getNumberOfTokens();

    /**
     * Return the number of tokens that are currently tested
     * by a transition, but that reside in this place.
     * We do not count multiple tests on a single token as
     * multiple tokens.
     */
    public abstract int getNumberOfTestedTokens();

    public boolean isEmpty() {
        return getNumberOfTokens() + getNumberOfTestedTokens() == 0;
    }

    public abstract int getTokenCount(Object token);

    public abstract TimeSet getFreeTimeSet(Object token);

    // Here consider only already tested tokens.
    public abstract boolean containsTestedToken(Object token);

    /**
     * The associated net instance will call this method while
     * its creation is confirmed and trace messages are printed.
     *
     * I will notify the database about all tokens that I kept during
     * my initialization, if the tokens must be available early.
     *
     * You must call this method at most once.
     *
     * In fact, this method is rather a kludge in the sense
     * that transaction support should not be activated
     * at all if a net requests early confirmation.
     */
    void earlyConfirmation() {
        // Do nothing.
    }

    /**
     * My net will call this method while its creation is traced.
     *
     */
    void earlyConfirmationTrace(StepIdentifier stepIdentifier) {
        // Do nothing.
    }

    /**
     * Upon creation, no transition will be checked for enabledness
     * by a searcher, because it is not contained in the
     * search queue of possibly activated transitions.
     * By calling this method, all transitions are inserted into the search
     * queue. Then all tokens of the initial marking that might
     * have been held back are inserted into the places.
     *
     * Additionally, if the net's automatic registration flag is
     * set, the net will be registered at the net list.
     *
     * @see de.renew.net.Net#setAutomaticRegistration(boolean)
     * @see de.renew.net.NetInstanceList
     *
     */
    void lateConfirmation(StepIdentifier stepIdentifier) {
        // Do nothing.
    }

    /**
     * Deserialization method, behaves like default readObject
     * method except restoring additional transient fields.
     * Creates new lock object.
     **/
    private void readObject(ObjectInputStream in)
            throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        lock = new Lock();
        listeners = new PlaceEventListenerSet();
    }
}