package de.renew.ant;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.taskdefs.Mkdir;
import org.apache.tools.ant.types.FileSet;

import java.io.File;


/**
 * Enhances the <code>JavaCC</code> task delivered with Ant to
 * work on a file set. The generated Java source of all processed
 * grammars can be put in a destination directory tree similar to
 * the one where the grammar files are kept.
 *
 * @author "Timo Carl" <6carl@informatik.uni-hamburg.de>
 *
 * @see org.apache.tools.ant.taskdefs.optional.javacc.JavaCC
 **/
public abstract class AbstractEnhancedJavaCCTask extends Task {
    protected File javacchome;
    protected File destdir;
    protected FileSet fileset;
    protected File srcbase;
    protected String singleTaskName;

    public AbstractEnhancedJavaCCTask(String singleTaskName) {
        this.singleTaskName = singleTaskName;
    }

    @Override
    public void init() throws BuildException {
        super.init();
    }

    public void setDestdir(File destdir) {
        this.destdir = destdir;
    }

    public void setJavacchome(File javacc) {
        this.javacchome = javacc;
    }

    public void addConfiguredFileset(FileSet fileset) {
        this.fileset = fileset;
        this.srcbase = fileset.getDir(getProject());
    }

    public void setSrcbase(File srcbase) {
        this.srcbase = srcbase;
    }

    @Override
    public void execute() throws BuildException {
        DirectoryScanner ds = fileset.getDirectoryScanner(getProject());
        String[] javaccFiles = ds.getIncludedFiles();
        File basedir = ds.getBasedir();
        for (int i = 0; i < javaccFiles.length; i++) {
            File f = new File(basedir.getAbsolutePath() + File.separator
                            + javaccFiles[i]);
            if (!f.isFile()) {
                throw new BuildException(
                                "javacc-File " + f + " does not exist!");
            }
            mkdirAndExecuteJavaCC(f);
        }
    }

    protected void mkdirAndExecuteJavaCC(File javaccFile)
                    throws BuildException {
        File directory = getGenerationDirectory(javaccFile);

        // Create dir if not exist
        if (!directory.isDirectory()) {
            Mkdir mkdir = (Mkdir) getProject().createTask("mkdir");
            mkdir.setDir(directory);
            mkdir.init();
            mkdir.execute();
        }
        executeJavaCC(javaccFile, directory);
    }

    protected abstract void executeJavaCC(File javaccFile, File directory)
                    throws BuildException;

    protected File getGenerationDirectory(File javaccFile) {
        String targetDir = javaccFile.getParent();
        String packageDir = targetDir
                        .substring(srcbase.getAbsolutePath().length());
        File directory = new File(destdir.getAbsolutePath() + File.separator
                        + packageDir);
        return directory;
    }
}