package de.renew.rnrg.gui;

import CH.ifa.draw.figures.EllipseFigure;
import CH.ifa.draw.figures.TextFigure;

import CH.ifa.draw.framework.DrawingView;

import CH.ifa.draw.util.StorableInput;
import CH.ifa.draw.util.StorableOutput;

import de.renew.gui.GuiPlugin;
import de.renew.gui.MultipleTokenFigure;

import de.renew.net.NetInstance;
import de.renew.net.Place;
import de.renew.net.PlaceInstance;

import de.renew.remote.MarkingAccessor;
import de.renew.remote.MarkingAccessorImpl;
import de.renew.remote.NetInstanceAccessorImpl;
import de.renew.remote.ObjectAccessor;

import de.renew.rnrg.elements.Node;

import de.renew.util.Base64Coder;

import java.awt.Color;
import java.awt.Point;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import java.rmi.RemoteException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;


public class ExploredNodeFigure extends EllipseFigure {
    private static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                    .getLogger(ExploredNodeFigure.class);
    Node node;

    /**
     * For loading as a Storable.
     */
    public ExploredNodeFigure() {
    }

    public ExploredNodeFigure(Node node) {
        super(new Point(0, 0), new Point(20, 20));
        setFillColor(Color.WHITE);
        this.node = node;
    }

    @Override
    public boolean inspect(DrawingView view, boolean alternate) {
        if (node != null) {
            NetInstance instance = node.getRootNetInstance();
            try {
                // RemotePlugin.wrapInstance can not be used without a running simulation.
                // This is fine since it would wrongly associate the net instance with the
                // current simulation anyway.
                //
                // Therefore we construct the accessor ourselves and set the simulation
                // environment to null.
                //
                // It could be set to the environment that was present when the graph was
                // constructed instead, but that should not be necessary and it is probably
                // not save to use functionality of the Remote plug-in that depends on the
                // environment anyway.
                GuiPlugin.getCurrent().openInstanceDrawing(
                                new NetInstanceAccessorImpl(instance, null));
            } catch (RemoteException e) {
                throw new RuntimeException(e);
            }

            return true;
        } else {
            if (logger.isDebugEnabled()) {
                logger.debug("Net state seems to have been lost.");
            }
            return false;
        }
    }

    @Override
    public void write(StorableOutput dw) {
        super.write(dw);

        // Store contained net state in serialized form inside a string:
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ObjectOutputStream oos;
        IOException serializeE = null;
        try {
            oos = new ObjectOutputStream(baos);
            try {
                oos.writeObject(node);
            } catch (IOException e) {
                serializeE = e;
            } finally {
                oos.close();
            }
        } catch (IOException e) {
            serializeE = e;
        }

        if (serializeE == null) {
            dw.writeString(Base64Coder.encode(baos.toByteArray()));
        } else {
            logger.warn(ExploredNodeFigure.class.getSimpleName()
                            + ": could not serialize internal net state: ",
                            serializeE);
            dw.writeString("");
        }
    }

    @Override
    public void read(StorableInput dr) throws IOException {
        super.read(dr);

        // Read contained net state in serialized form from a string:
        String nodeStr = dr.readString();
        if (nodeStr == "") {
            logger.warn(ExploredNodeFigure.class.getSimpleName()
                            + ": serialized internal net state is missing.");
        } else {
            byte[] data = Base64Coder.decode(nodeStr);
            ObjectInputStream ois = new ObjectInputStream(
                            new ByteArrayInputStream(data));

            try {
                node = (Node) ois.readObject();
            } catch (ClassNotFoundException | IOException e) {
                final String desc = ExploredNodeFigure.class.getSimpleName()
                                + ": could not load serialized internal state:";
                if (logger.isDebugEnabled()) {
                    // long version
                    logger.error(desc, e);
                } else {
                    // short version
                    logger.error(desc + " " + e);
                }
            } finally {
                ois.close();
            }
        }
    }

    /*
     *  Code from {@link de.renew.gui.TokenBagFigure#toString()}
     */
    static private void describePlaceInstance(StringBuffer output,
                                              PlaceInstance placeInstance) {
        output.append('{');

        try {
            MarkingAccessor marking = new MarkingAccessorImpl(placeInstance,
                            null);

            int distinctTokenCount = marking.getDistinctTokenCount();
            for (int i = 0; i < distinctTokenCount; i++) {
                ObjectAccessor token = marking.getToken(i);
                int mult = marking.getTokenFreeCount(i);
                boolean isTested = marking.getTokenTested(i);

                output.append(MultipleTokenFigure.getMultString(mult,
                                isTested));
                output.append(((token == null) ? "null" : token.asString()));

                if (i < distinctTokenCount - 1) {
                    output.append(", ");
                }
            }

            output.append('}');
        } catch (RemoteException e) {
            logger.error(e.getMessage(), e);
        }
    }

    static private void describeNetInstance(StringBuffer descBuffer,
                                            NetInstance inst) {
        List<Place> places = new ArrayList<Place>(inst.getNet().places());
        // Sort by place name.
        Collections.sort(places, new Comparator<Place>() {
            @Override
            public int compare(Place p1, Place p2) {
                return p1.getName().compareTo(p2.getName());
            }
        });

        Iterator<Place> iter = places.iterator();

        while (iter.hasNext()) {
            final Place place = iter.next();

            descBuffer.append(place.getName());
            descBuffer.append('=');
            describePlaceInstance(descBuffer, inst.getInstance(place));

            if (iter.hasNext()) {
                descBuffer.append(", ");
            }
        }
    }

    static private String describeNode(StringBuffer nodeDescBuffer, Node node) {
        NetInstance root = node.getRootNetInstance();
        List<NetInstance> others = new ArrayList<NetInstance>(
                        node.getNetInstances());
        // Sort by instance name.
        Collections.sort(others, new Comparator<NetInstance>() {
            @Override
            public int compare(NetInstance i1, NetInstance i2) {
                return i1.getID().compareTo(i2.getID());
            }
        });

        StringBuffer rootDescBuffer = new StringBuffer();
        describeNetInstance(rootDescBuffer, root);

        nodeDescBuffer.append(root);
        nodeDescBuffer.append(": ");
        nodeDescBuffer.append(rootDescBuffer);

        for (NetInstance inst : others) {
            if (inst != root) {
                nodeDescBuffer.append('\n');
                nodeDescBuffer.append(inst);
                nodeDescBuffer.append(": ");
                describeNetInstance(nodeDescBuffer, inst);
            }
        }

        return rootDescBuffer.toString();
    }

    private TextFigure createTextFigure(String text) {
        TextFigure fig = new TextFigure(text);
        fig.setReadOnly(true);
        fig.setParent(this);
        // Inform text of changes to this figure.
        addToContainer(fig);
        return fig;
    }

    public TextFigure createTextFigure(int inscMode) {
        if (inscMode == GraphDrawing.INSC_DEPTH) {
            return createTextFigure(String.valueOf(node.getDepth()));
        } else {
            StringBuffer nodeDescBuffer = new StringBuffer();
            String rootDesc = describeNode(nodeDescBuffer, node);

            return new Inscription(rootDesc, nodeDescBuffer.toString(),
                            inscMode == GraphDrawing.INSC_ALL, this);
        }
    }
}