package de.renew.rnrg.elements;

import de.renew.engine.searcher.Searcher;
import de.renew.engine.simulator.Binding;
import de.renew.engine.simulator.GraphFinder;
import de.renew.engine.simulator.GraphFinder.TransitionChecker;

import de.renew.net.Net;
import de.renew.net.NetInstance;
import de.renew.net.ResettableNetInstance;
import de.renew.net.Transition;

import de.renew.unify.Impossible;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;


/**
 * A node in the reachability graph
 * representing the instances of a root net instance and all contained net instances.
 *
 * @author Michael Simon
 *
 */
public class NodeImpl extends Node {
    private static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                    .getLogger(NodeImpl.class);
    private static final long serialVersionUID = 0L;
    private final NetInstanceState rootInstanceState;
    private final Set<ResettableNetInstance> instances;
    private final long depth; // Minimal distance from the start node in the graph.

    public NodeImpl(Set<ResettableNetInstance> instances,
                    NetInstanceState rootInstanceState) {
        this(instances, rootInstanceState, 0);
    }

    private NodeImpl(Set<ResettableNetInstance> instances,
                     NetInstanceState rootInstanceState, long depth) {
        this.rootInstanceState = rootInstanceState;
        this.instances = instances;
        this.depth = depth;
    }

    public NodeImpl(Net net) {
        rootInstanceState = new NetInstanceState(net);
        instances = Collections.singleton(rootInstanceState.instance);
        depth = 0;
    }

    @Override
    public int hashCode() {
        return rootInstanceState.hashCode();
    }

    @Override
    public boolean equals(Object obj) {
        return (obj instanceof NodeImpl) && rootInstanceState
                        .equals(((NodeImpl) obj).rootInstanceState);
    }

    /**
     * Ensure that this state is in the map.
     * Add this node to the map and the search queue,
     * if an equal node is not already present in the map.
     *
     * @param nodes the map
     * @param unexploredNodes the search queue
     * @return the equal node contained in the map or {@code null}
     */
    private Node addIfNonexistent(final Map<NetInstanceState, Node> nodes,
                                  final Collection<NodeImpl> unexploredNodes) {
        // Get the corresponding node from the map,
        // if one already exists:
        final Node node = nodes.get(rootInstanceState);
        if (node != null) {
            return node;
        } else {
            // Add this new node to the map:
            addTo(nodes);
            // Add it to the queue so it gets explored later:
            unexploredNodes.add(this);

            return this;
        }
    }

    public void addTo(final Map<NetInstanceState, Node> nodes) {
        nodes.put(rootInstanceState, this);
    }

    /**
     * Create a new {@link NodeImpl} object that is a copy of this object.
     * Drop net instances that are unreachable from the root net instance.
     * Increase the depth field.
     *
     * @return the new Node object
     */
    private NodeImpl successorFromCurrentState() {
        Map<NetInstance, ResettableNetInstance> copyMap = new HashMap<NetInstance, ResettableNetInstance>(
                        instances.size());

        // Initialize copyMap by creating a new ResettableNetInstance for every existing state:
        for (NetInstance instance : instances) {
            try {
                copyMap.put(instance, new ResettableNetInstance(
                                instance.getNet(), false));
            } catch (Impossible e) {
                throw new RuntimeException(e);
            }
        }

        NetInstance rootNetInstance = rootInstanceState.getNetInstance();
        Set<NetInstance> explored = new HashSet<NetInstance>();

        {
            Set<NetInstance> toExplore = new HashSet<NetInstance>();
            toExplore.add(rootNetInstance);

            while (!toExplore.isEmpty()) {
                final NetInstance exploring;
                // Pop an element from toExplore:
                {
                    Iterator<NetInstance> iter = toExplore.iterator();
                    exploring = iter.next();
                    iter.remove();
                }
                explored.add(exploring);


                // Set the new net instance for exploring to the present value:
                Set<NetInstance> newToExplore = copyMap.get(exploring)
                                .setTo(exploring, copyMap);
                // Only add net instances to toExplore that are not already explored:
                newToExplore.removeAll(explored);
                toExplore.addAll(newToExplore);
            }
        }

        // The copy of the root net state.
        NetInstanceState newRootNetState = new NetInstanceState(
                        copyMap.get(rootNetInstance));


        // Only take net instances that were actually explored:
        Set<ResettableNetInstance> newInstances = new HashSet<ResettableNetInstance>(
                        explored.size());
        for (NetInstance exp : explored) {
            newInstances.add(copyMap.get(exp));
        }

        return new NodeImpl(newInstances, newRootNetState, depth + 1);
    }

    /**
     * Explore the given binding and reset this node to the given backup state.
     * Create and return a new {@link NodeImpl} object corresponding to the state after the binding execution.
     *
     * @param binding the binding to execute
     * @param backup the state to reset this node to
     * @return the newly created successor {@link NodeImpl}
     */
    private NodeImpl exploreBinding(Binding binding,
                                    Map<NetInstance, ResettableNetInstance> backup) {
        // Execute a binding directly on the net instances of this node.
        // The effect will be reset later.
        binding.execute(null, false);

        // The unreachable net instances are dropped in the copy method.
        // It also converts all NetInstance objects to ResettableNetInstance objects
        // which is needed when accessing the nodes map.
        NodeImpl successor = successorFromCurrentState();

        // Reset this node to its original state which it represents.
        resetToBackup(backup);

        return successor;
    }

    public boolean exploreEdges(long maxDepth, TransitionChecker checker,
                                Map<NetInstanceState, Node> nodes,
                                Collection<NodeImpl> unexploredNodes) {
        boolean maxDepthReached = false;
        edges = new HashSet<Edge>();
        Map<NetInstance, ResettableNetInstance> backup = createBackup();

        if (maxDepth == depth) {
            // Depth limit reached: don't allow any further exploration.
            checker = new TransitionChecker() {
                @Override
                public boolean isExplorable(Transition trans) {
                    return false;
                }
            };
            maxDepthReached = true;
        }

        for (NetInstance instance : instances) {
            for (Transition transition : instance.getNet().transitions()) {
                final String name;
                if (instance == rootInstanceState.instance) {
                    name = transition.toString();
                } else {
                    name = instance.getInstance(transition).toString();
                }

                GraphFinder finder = new GraphFinder(checker);
                new Searcher().searchAndRecover(finder,
                                instance.getInstance(transition), null);

                for (Binding binding : finder.unexplorableBindings()) {
                    final String description = binding.getDescription();

                    if (logger.isDebugEnabled()) {
                        logger.debug("not exploring: " + description);
                    }

                    edges.add(new InscribedEdge(name, description, null));
                }

                for (Binding binding : finder.explorableBindings()) {
                    final String description = binding.getDescription();

                    if (logger.isDebugEnabled()) {
                        logger.debug("exploring: " + description);
                    }

                    // Explore binding and get the corresponding node from the map,
                    // if one already exists and add the new node otherwise:
                    final Node target = exploreBinding(binding, backup)
                                    .addIfNonexistent(nodes, unexploredNodes);

                    target.addPredecessor(this);
                    edges.add(new InscribedEdge(name, description, target));
                }
            }
        }
        return maxDepthReached;
    }

    private Map<NetInstance, ResettableNetInstance> createBackup() {
        Map<NetInstance, ResettableNetInstance> backup = new HashMap<NetInstance, ResettableNetInstance>(
                        instances.size());

        for (ResettableNetInstance instance : instances) {
            backup.put(instance, instance.copy());
        }

        return backup;
    }

    private void resetToBackup(Map<NetInstance, ResettableNetInstance> backup) {
        for (ResettableNetInstance instance : instances) {
            instance.setTo(backup.get(instance));
        }
    }

    @Override
    public Collection<? extends NetInstance> getNetInstances() {
        return instances;
    }

    @Override
    public NetInstance getRootNetInstance() {
        return rootInstanceState.getNetInstance();
    }

    @Override
    public long getDepth() {
        return depth;
    }
}