/**
 *
 */
package de.renew.rnrg.commands;

import CH.ifa.draw.DrawPlugin;

import CH.ifa.draw.util.Command;

import de.renew.net.TransitionCheckerImpl;

import de.renew.rnrg.elements.Graph.Aborter;
import de.renew.rnrg.gui.GraphDrawing;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import java.text.ParseException;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.InputVerifier;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JFormattedTextField.AbstractFormatter;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;


/** Create reachability graph of a reference net with unification.
 *
 * @author Michael Simon
 *
 */
public class ReachabilityGraphCommand extends Command implements Runnable {
    public static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                    .getLogger(ReachabilityGraphCommand.class);
    public static final String CMD_NAME = "Ref Net Reachability Graph...";
    private static final String FRAME_NAME = "Graph Creation";
    private JFrame frame;

    /*
     * For ReachabilityGraphClCommand.createGraph(...).
     */
    private boolean action = true;
    private boolean creation = false;
    private int inscMode;
    private boolean limitDepth = false;
    private long maxDepth = 16;
    private GUIAborter aborter;

    private class GUIAborter implements Aborter {
        private boolean abort = false;

        void initiateAbort() {
            abort = true;
        }

        @Override
        public boolean abort() {
            return abort;
        }
    }

    public ReachabilityGraphCommand() {
        super(CMD_NAME);
    }

    /**
     * @see CH.ifa.draw.util.Command#execute()
     */
    @Override
    public void execute() {
        createFrame();
    }

    private void start() {
        new Thread(this).start();
    }

    private void createFrame() {
        // Make sure this method is done before closeFrame() is called.
        frame = new JFrame(FRAME_NAME);
        aborter = new GUIAborter();

        /*
         * Create check/combo boxes and text field for the parameters pane.
         */
        final JCheckBox actionBox = new JCheckBox("Skip Action Execution",
                        action);
        actionBox.addItemListener(new ItemListener() {
            @Override
            public void itemStateChanged(ItemEvent e) {
                action = e.getStateChange() == ItemEvent.SELECTED;
            }
        });
        actionBox.setToolTipText(
                        "Do not explore bindings that would fire transitions with action inscriptions.");

        final JCheckBox creationBox = new JCheckBox(
                        "Skip Net Instance Creation", creation);
        creationBox.addItemListener(new ItemListener() {
            @Override
            public void itemStateChanged(ItemEvent e) {
                creation = e.getStateChange() == ItemEvent.SELECTED;
            }
        });
        creationBox.setToolTipText(
                        "Do not explore bindings that would create new net instances.");

        final JComboBox<String> inscBox = new JComboBox<String>(
                        GraphDrawing.INSC_MODES);
        inscBox.setSelectedIndex(inscMode);
        inscBox.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                inscMode = inscBox.getSelectedIndex();
            }
        });
        inscBox.setToolTipText("What to inscribe to graph nodes.");

        final JFormattedTextField depthField = new JFormattedTextField(
                        maxDepth);
        depthField.setInputVerifier(new InputVerifier() {

            /**
             * Verify that the input is a positive Long value.
             */
            @Override
            public boolean verify(JComponent input) {
                AbstractFormatter formatter = depthField.getFormatter();
                try {
                    Object newValue = formatter
                                    .stringToValue(depthField.getText());
                    if (newValue instanceof Long && ((Long) newValue) >= 0) {
                        return true;
                    }
                } catch (ParseException e) {
                    // Do nothing.
                }
                return false;
            }
        });
        depthField.addPropertyChangeListener(new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                maxDepth = (Long) depthField.getValue();
            }
        });
        depthField.setEnabled(limitDepth);
        depthField.setHorizontalAlignment(SwingConstants.TRAILING);
        depthField.setToolTipText(
                        "Maximum depth up to which graph nodes are explored.");

        final JCheckBox depthBox = new JCheckBox("Depth Limit", limitDepth);
        depthBox.addItemListener(new ItemListener() {
            @Override
            public void itemStateChanged(ItemEvent e) {
                limitDepth = e.getStateChange() == ItemEvent.SELECTED;
                depthField.setEnabled(limitDepth);
            }
        });
        depthBox.setToolTipText(
                        "Stop exploring graph nodes after reaching this depth.");

        /*
         * Create the parameters pane with check/combo boxes and text field.
         */
        JPanel paramPane = new JPanel();
        paramPane.setLayout(new BoxLayout(paramPane, BoxLayout.Y_AXIS));
        paramPane.add(actionBox);
        paramPane.add(creationBox);

        paramPane.add(Box.createRigidArea(new Dimension(0, 5)));

        JPanel inscPane = new JPanel();
        inscPane.setLayout(new BoxLayout(inscPane, BoxLayout.Y_AXIS));
        inscPane.setBorder(BorderFactory.createTitledBorder(
                        BorderFactory.createEmptyBorder(),
                        "Node Inscriptions"));
        inscPane.add(inscBox);
        // Restrict the inscription box's maximum size.
        limitMaxHeight(inscBox);
        inscBox.setAlignmentX(Component.LEFT_ALIGNMENT);
        paramPane.add(inscPane);

        JPanel depthPane = new JPanel();
        depthPane.setLayout(new BoxLayout(depthPane, BoxLayout.X_AXIS));
        depthPane.add(depthBox);
        depthPane.add(Box.createRigidArea(new Dimension(5, 0)));
        depthPane.add(depthField);
        depthPane.add(Box.createRigidArea(new Dimension(5, 0)));
        // Restrict the depth field's maximum size.
        limitMaxHeight(depthField);
        //depthField.setColumns(6);
        depthPane.setAlignmentX(Component.LEFT_ALIGNMENT);
        paramPane.add(depthPane);

        paramPane.add(Box.createRigidArea(new Dimension(0, 5)));

        /*
         * Create buttons for the button pane.
         */
        final JButton startButton = new JButton("Start");
        startButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                startButton.setEnabled(false);
                actionBox.setEnabled(false);
                creationBox.setEnabled(false);
                start();
            }
        });
        // Make this the default button.
        frame.getRootPane().setDefaultButton(startButton);
        startButton.setToolTipText("Start the reachability graph creation.");

        final Action cancelAction = new AbstractAction("Cancel") {
            @Override
            public void actionPerformed(ActionEvent e) {
                closeFrame();
            }
        };
        JButton cancelButton = new JButton(cancelAction);
        cancelButton.setToolTipText(
                        "Cancel reachability graph creation and close window.");

        // Closing the frame or pressing ESC is the same as clicking Cancel.
        cancelButton.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(
                        KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
                        "cancel");
        cancelButton.getActionMap().put("cancel", cancelAction);
        frame.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent w) {
                closeFrame();
            }
        });

        /*
         * Create the button pane.
         */
        JPanel buttonPane = new JPanel(new GridLayout(1, 0));
        buttonPane.add(cancelButton);
        buttonPane.add(startButton);

        /*
         * Add the content to the frame.
         */
        Container contentPane = frame.getContentPane();
        contentPane.setLayout(new BoxLayout(contentPane, BoxLayout.Y_AXIS));

        paramPane.setAlignmentX(Component.CENTER_ALIGNMENT);
        buttonPane.setAlignmentX(Component.CENTER_ALIGNMENT);
        // Restrict the button pane's maximum size.
        buttonPane.setMaximumSize(buttonPane.getPreferredSize());

        contentPane.add(Box.createGlue());
        contentPane.add(paramPane);
        contentPane.add(Box.createGlue());
        contentPane.add(buttonPane);

        /*
         * Restrict sizes.
         */
        frame.pack();
        setMaxWidth(inscPane, creationBox.getSize().width);
        setMaxWidth(depthPane, creationBox.getSize().width);

        /*
         * Make the frame visible.
         */


        // Receive the focus.
        frame.setAlwaysOnTop(true);
        frame.pack();
        // Focus on the start button.
        startButton.requestFocusInWindow();
        DrawPlugin.getCurrent().getMenuManager().getWindowsMenu()
                        .addFrame(DrawPlugin.WINDOWS_CATEGORY_TOOLS, frame);
        // It is important that this is done last so closeFrame() can not be called before.
        frame.setVisible(true);
    }

    private static void limitMaxHeight(JComponent component) {
        component.setMaximumSize(new Dimension(component.getMaximumSize().width,
                        component.getPreferredSize().height));
    }

    private static void setMaxWidth(JComponent component, int width) {
        component.setMaximumSize(new Dimension(width,
                        component.getMaximumSize().height));

    }

    private synchronized void closeFrame() {
        if (frame != null) {
            aborter.initiateAbort();
            frame.setVisible(false);
            DrawPlugin.getCurrent().getMenuManager().getWindowsMenu()
                            .removeFrame(frame);
            frame = null;
            aborter = null;
        }
    }

    @Override
    public void run() {
        String str = ReachabilityGraphClCommand.createGraph(null, inscMode,
                        limitDepth ? maxDepth : -1,
                        new TransitionCheckerImpl(action, creation), aborter);

        closeFrame();

        if (!str.isEmpty()) {
            logger.error("creating reachability graph: " + str);
        }
    }
}