/**
 *
 */
package de.renew.rnrg.commands;

import CH.ifa.draw.DrawPlugin;

import CH.ifa.draw.application.DrawApplication;

import CH.ifa.draw.io.StatusDisplayer;

import de.renew.application.SimulatorPlugin;

import de.renew.engine.simulator.GraphFinder.TransitionChecker;

import de.renew.gui.CPNDrawing;

import de.renew.net.Net;
import de.renew.net.NetNotFoundException;
import de.renew.net.TransitionCheckerImpl;

import de.renew.plugin.command.CLCommand;

import de.renew.rnrg.elements.Graph;
import de.renew.rnrg.elements.Graph.Aborter;
import de.renew.rnrg.elements.NodeImpl;
import de.renew.rnrg.gui.GraphDrawing;

import java.io.PrintStream;


/** Create reachability graph of a Java reference net.
 *
 * @author Michael Simon
 *
 */
public class ReachabilityGraphClCommand implements CLCommand, StatusDisplayer {
    public static final String CMD = "reachgraph";

    /**
     * @see de.renew.plugin.command.CLCommand#execute(java.lang.String[], java.io.PrintStream)
     */
    @Override
    public void execute(String[] args, PrintStream response) {
        String name;
        if (args.length == 0) {
            name = null;
        } else {
            name = args[0];
        }

        showStatus(createGraph(name, GraphDrawing.INSC_NOTHING));
    }

    /** Create and open the reachability graph for the net with the given name.
     * @param name - a name of a renew net
     * @param inscMode - what should be inscribed to the nodes?
     * @return the type of error or an empty string
     */
    static protected String createGraph(String name, int inscMode) {
        return createGraph(name, inscMode, -1,
                        new TransitionCheckerImpl(true, false), new Aborter() {
                            @Override
                            public boolean abort() {
                                return false;
                            }
                        });
    }

    static protected String createGraph(String name, int inscMode,
                                        long maxDepth,
                                        TransitionChecker checker,
                                        Aborter aborter) {
        DrawApplication gui = DrawPlugin.getGui();
        if (gui != null && gui.drawing() instanceof CPNDrawing) {
            // Compile cpn drawing and return printGraph(Net).
            if (name == null) {
                name = gui.drawing().getName();
            }

            // Lock the SimulatorPlugin so the known nets can not change.
            SimulatorPlugin.lock.lock();
            try {
                SimulatorPlugin sim = SimulatorPlugin.getCurrent();

                // Whether a new simulation is set up for us
                // and thus must be terminated afterwards.
                boolean newSimulation = sim.getCurrentEnvironment() == null;

                // If no simulation is already set up...
                if (newSimulation) {
                    // ...Set up a new simulation.
                    sim.setupSimulation(null);
                }

                try {
                    Net net = Net.forName(name);

                    // Make sure the initial token get inserted.
                    net.setEarlyTokens(true);

                    // Create the actual graph and open it.
                    createMultiGraph(net, inscMode, maxDepth, checker, aborter);
                } catch (NetNotFoundException e) {
                    return "net " + name + " not found";
                } finally {
                    if (newSimulation) {
                        // Terminate simulation that was set up above.
                        sim.terminateSimulation();
                    }
                }
            } finally {
                SimulatorPlugin.lock.unlock();
            }

            return "";
        } else {
            return "no net name given and no cpn drawing open in GUI";
        }
    }

    /** Create and open the reachability graph.
     * @param net - a renew net
     * @param checker - the TransitionChecker to use
     * @param aborter - can abort the graph creation
     * @param inscMode - what should be inscribed to the nodes?
     * @param maxDepth - the maximal depth to explore
     * @return the String that presents the reachability graph of the net.
     */
    static protected void createMultiGraph(Net net, int inscMode, long maxDepth,
                                           TransitionChecker checker,
                                           Aborter aborter) {
        NodeImpl startNode = new NodeImpl(net);
        Graph graph = new Graph(startNode, maxDepth, checker, aborter);

        if (!aborter.abort()) {
            DrawPlugin.getGui().openDrawing(new GraphDrawing(graph, inscMode));
        }
    }

    /**
     * @see de.renew.plugin.command.CLCommand#getDescription()
     */
    @Override
    public String getDescription() {
        return "Create and open reachability graph of a reference net.";
    }

    /**
     * @see CH.ifa.draw.io.StatusDisplayer#showStatus(java.lang.String)
     */
    @Override
    public void showStatus(String message) {
        System.out.println(message);
    }

    @Override
    public String getArguments() {
        return "[netName]";
    }
}