package de.renew.net;

import de.renew.unify.Impossible;

import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;


public class ResettableNetInstance extends NetInstanceImpl {
    static final int HASH_CODE_MAX_DEPTH = 4;
    String originalID;

    public ResettableNetInstance(Net net, boolean wantInitialTokens)
                    throws Impossible {
        super(net, wantInitialTokens);
        originalID = super.getID();
    }

    public ResettableNetInstance(NetInstance other) throws Impossible {
        this(other, null);
    }

    public ResettableNetInstance(NetInstance other,
                                 Map<NetInstance, ResettableNetInstance> copyMap)
                    throws Impossible {
        super(other.getNet(), false);
        setOriginalID(other);
        setTo(other, copyMap);
    }

    private void setOriginalID(NetInstance other) {
        if (other instanceof ResettableNetInstance) {
            originalID = ((ResettableNetInstance) other).originalID;
        } else {
            originalID = other.getID();
        }
    }

    @Override
    public String getID() {
        return originalID + "," + super.getID();
    }

    @Override
    public String toString() {
        return getNet() + "[" + originalID + "]";
    }

    public ResettableNetInstance copy() {
        return copy(null, null);
    }

    public ResettableNetInstance copy(Map<NetInstance, ResettableNetInstance> copyMap,
                                      Collection<NetInstance> containedNetIntances) {
        try {
            return new ResettableNetInstance(this, copyMap);
        } catch (Impossible e) {
            throw new RuntimeException(e);
        }
    }

    public Collection<NetInstance> setTo(NetInstance other) {
        return setTo(other, null);
    }

    public Set<NetInstance> setTo(NetInstance other,
                                  Map<NetInstance, ResettableNetInstance> copyMap) {
        assert getNet() == other.getNet();
        if (other == this) {
            return null;
        } else {
            Set<NetInstance> containedNetInstances = new HashSet<NetInstance>();

            setOriginalID(other);

            for (Place place : getNet().places) {
                MultisetPlaceInstance myInst = (MultisetPlaceInstance) getInstance(
                                place);
                PlaceInstance otInst = other.getInstance(place);

                // empty my place instance
                myInst.extractAllTokens(null, null);


                // get all token in the other's place instance
                Set<Object> tokens = otInst.getDistinctTestableTokens();

                for (Object token : tokens) {
                    // how many times is this token present?
                    int count = otInst.getTokenCount(token);

                    if (token == other) {
                        // Store this net instance,
                        // if the token is the other net instance.
                        // This preserves the self-reference.
                        token = this;
                    } else if (token instanceof NetInstance) {
                        if (copyMap != null) {
                            NetInstance instance = (NetInstance) token;
                            containedNetInstances.add(instance);
                            ResettableNetInstance replacement = copyMap
                                            .get(instance);

                            if (replacement == null) {
                                try {
                                    replacement = new ResettableNetInstance(
                                                    instance);
                                } catch (Impossible e) {
                                    throw new RuntimeException(e);
                                }
                                copyMap.put(instance, replacement);
                            }

                            token = replacement;
                        }
                    }

                    // insert it at my place instance as many times (at time 0)
                    myInst.internallyInsertTokenMultiple(token, 0.0, count,
                                    false);
                }
            }

            return containedNetInstances;
        }
    }

    public int stateHashCode() {
        return stateHashCode(HASH_CODE_MAX_DEPTH);
    }

    private int stateHashCode(int depth) {
        int h = 0;
        for (Place place : getNet().places) {
            h += getPlaceHash(place, depth);
        }

        return h;
    }

    private int getPlaceHash(Place place, int depth) {
        int h = 0;
        PlaceInstance inst = getInstance(place);
        for (Object token : inst.getDistinctTokens()) {
            int tokenHash;
            if (!(token instanceof NetInstance)) {
                tokenHash = token.hashCode();
            } else if (token == this) {
                tokenHash = -42;
            } else if (depth > 0) {
                // Every token that is a NetInstance should also be a ResettableNetInstance
                // at this point.
                if (!(token instanceof ResettableNetInstance)) {
                    throw new ClassCastException(token + " should have been a "
                                    + ResettableNetInstance.class.getName()
                                    + " object");
                }
                tokenHash = ((ResettableNetInstance) token)
                                .stateHashCode(depth - 1);
            } else {
                tokenHash = 0;
            }

            h += tokenHash ^ inst.getTokenCount(token);
        }

        return h ^ place.hashCode();
    }

    /**
     * Whether this net instance has the same state as obj.
     *
     * obj must be a ResettableNetInstance object.
     * There has to be a map from all net instances
     * (recursively) contained in this net instance
     * to the net instances in the other net instance
     * such that all net instances have the same state
     * as the one they are mapped to.
     *
     * If true is returned, equalityMap contains a mapping from this to obj.
     * If false is returned, equalityMap does not contain that mapping.
     *
     * @param obj
     * @param equalityMap
     * @return whether the states are equal
     */
    public boolean stateEquals(final Object obj,
                               final Map<ResettableNetInstance, ResettableNetInstance> equalityMap) {
        if (obj == this) {
            return true;
        }

        if (!(obj instanceof NetInstance)) {
            return false;
        } else if (!(obj instanceof ResettableNetInstance)) {
            throw new RuntimeException(
                            "A " + ResettableNetInstance.class.getSimpleName()
                                            + " object was compared with a "
                                            + NetInstance.class.getSimpleName()
                                            + " object that is not a "
                                            + ResettableNetInstance.class
                                                            .getSimpleName()
                                            + ".");
        }

        ResettableNetInstance other = (ResettableNetInstance) obj;
        if (getNet() != other.getNet()) {
            return false;
        }

        Set<Place> places = getNet().places;

        for (Place place : places) {
            if (!placesNormalTokensEqual(other, place)) {
                return false;
            }
        }

        equalityMap.put(this, other);
        if (!containedNetInstanceTokensEqual(other, places, equalityMap)) {
            equalityMap.remove(this);
            return false;
        }

        return true;
    }

    /**
     * Whether a single place contains the same normal tokens in this and the given net instance.
     * Normal tokens are all tokens that are not net instances.
     *
     * @param other the other net instance
     * @param place the place whose instances from both net instances are compared
     * @return
     */
    private boolean placesNormalTokensEqual(final NetInstance other,
                                            final Place place) {
        PlaceInstance myPlace = getInstance(place);
        PlaceInstance otherPlace = other.getInstance(place);

        Set<Object> myTokens = myPlace.getDistinctTokens();
        Set<Object> otherTokens = otherPlace.getDistinctTokens();

        // Both sets must have the same number of elements.
        if (myTokens.size() != otherTokens.size()) {
            return false;
        }

        for (Object token : myTokens) {
            if (!(token instanceof NetInstance)) {
                // Otherwise the other place must contain the exact same token.
                if (myPlace.getTokenCount(token) != otherPlace
                                .getTokenCount(token)) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Whether a map can be found from all net instances
     * (recursively) contained in this net instance
     * to the net instances in the given net instance
     * such that all net instances have the same state
     * as the ones they are mapped to.
     *
     * @param other the other net instance
     * @param places the places that have not been considered yet
     * @param equalityMap the map
     * @return
     */
    private boolean containedNetInstanceTokensEqual(final NetInstance other,
                                                    final Set<Place> places,
                                                    final Map<ResettableNetInstance, ResettableNetInstance> equalityMap) {
        if (!places.isEmpty()) {
            Set<Place> nextPlaces = new HashSet<Place>(places);
            Place place = nextPlaces.iterator().next();
            nextPlaces.remove(place);

            PlaceInstance myPlace = getInstance(place);
            PlaceInstance otherPlace = other.getInstance(place);

            Set<Object> myTokens = myPlace.getDistinctTokens();
            Set<Object> otherTokens = otherPlace.getDistinctTokens();

            return placesNetInstanceTokensEqual(other, myPlace, otherPlace,
                            myTokens, otherTokens, equalityMap, nextPlaces);
        } else {
            return true;
        }
    }

    private boolean placesNetInstanceTokensEqual(final NetInstance other,
                                                 final PlaceInstance myPlace,
                                                 final PlaceInstance otherPlace,
                                                 final Set<Object> myTokens,
                                                 final Set<Object> otherTokens,
                                                 final Map<ResettableNetInstance, ResettableNetInstance> equalityMap,
                                                 final Set<Place> nextPlaces) {
        HashSet<Object> nextTokens = new HashSet<Object>(myTokens);

        while (!nextTokens.isEmpty()) {
            Object token = nextTokens.iterator().next();
            nextTokens.remove(token);

            if (token instanceof ResettableNetInstance) {
                int myTokenCount = myPlace.getTokenCount(token);
                ResettableNetInstance netInst = (ResettableNetInstance) token;
                ResettableNetInstance otherNetInst = equalityMap.get(token);
                if (otherNetInst != null) {
                    if (myTokenCount != otherPlace
                                    .getTokenCount(otherNetInst)) {
                        return false;
                        // Break recursion.
                    }
                } else {
                    // The search has to try possible net instance matchings here.
                    for (Object otherToken : otherTokens) {
                        if (otherToken instanceof NetInstance && !equalityMap
                                        .values().contains(otherToken)
                        // Only consider this token, if it is not already mapped to.
                                        && myTokenCount == otherPlace
                                                        .getTokenCount(otherToken)
                                        && netInst.stateEquals(otherToken,
                                                        equalityMap)) {
                            // Notice that netInst.stateEquals updated equalityMap.
                            // Deepen the search here.
                            // Continue with the rest of the tokens of the place and the rest of the places.
                            if (placesNetInstanceTokensEqual(other, myPlace,
                                            otherPlace, nextTokens, otherTokens,
                                            equalityMap, nextPlaces)) {
                                return true;
                                // Break recursion.
                            } else {
                                equalityMap.remove(netInst);
                            }
                        }
                    }

                    return false;
                    // Break recursion.
                }
            } else if (token instanceof NetInstance) {
                throw new RuntimeException("A "
                                + ResettableNetInstance.class.getSimpleName()
                                + " object may not contain "
                                + NetInstance.class.getSimpleName()
                                + " tokens.");
            }
        }

        return containedNetInstanceTokensEqual(other, nextPlaces, equalityMap);
    }
}