package de.renew.navigator.vc;

import org.apache.log4j.Logger;

import java.io.File;

import java.util.HashMap;


/**
 * The abstract version control saves the used repository for a file in memory.
 *
 * @author Konstantin Simon Maria Moellers
 * @version 2015-10-19
 */
public abstract class AbstractVersionControl implements VersionControl {

    /**
     * Log4j logger instance.
     */
    public static final Logger logger = Logger
                    .getLogger(AbstractVersionControl.class);
    private static final HashMap<File, Repository> REPOSITORIES;

    static {
        REPOSITORIES = new HashMap<File, Repository>();
    }

    @Override
    final public boolean controls(File fileInRepository) {
        return REPOSITORIES.containsKey(fileInRepository)
                        || findRepository(fileInRepository) != null;
    }

    @Override
    final public Repository findRepository(File fileInRepository) {
        // Load cached variant for this exact file.
        if (REPOSITORIES.containsKey(fileInRepository)) {
            return REPOSITORIES.get(fileInRepository);
        }

        File rootDirectory = findRootDirectory(fileInRepository);
        if (rootDirectory == null) {
            logger.warn(fileInRepository + " belongs not to " + getName());
            return null;
        }
        // Load cached variant of the root directory.
        if (REPOSITORIES.containsKey(rootDirectory)) {
            Repository repository = REPOSITORIES.get(rootDirectory);
            REPOSITORIES.put(fileInRepository, repository);
            return repository;
        }
        
        final Repository repository = buildRepository(fileInRepository);

        if (repository == null) {
            logger.warn(fileInRepository + " belongs not to " + getName());
            return null;
        }

        // Cache the repository.
        REPOSITORIES.put(fileInRepository, repository);
        REPOSITORIES.put(repository.getRootDirectory(), repository);
        logger.debug("Found " + fileInRepository + ", belongs to " + getName());

        return repository;
    }

    @Override
    public String getName() {
        final String simpleName = getClass().getSimpleName();

        final int index = simpleName.indexOf("VersionControl");
        return index == -1 ? simpleName : simpleName.substring(0, index);
    }
    
    /**
     * Returns the root directory of a repository for a file in that repository.
     *      
     * @param fileInRepository The file in the repository
     * @return the root directory of the repository, <code>null</code> if the file is not in a repository 
     */
    abstract protected File findRootDirectory(File fileInRepository);

    abstract protected Repository buildRepository(File fileInRepository);


    /**
     * Displays an error for a given exception.
     *
     * @param exception the exception to display.
     */
    protected void errorException(Exception exception, String action) {
        logger.error("Error while " + action, exception);
        if (logger.isDebugEnabled()) {
            logger.debug(getClass().getSimpleName() + ": " + exception);
        }
    }
}