/**
 *
 */
package de.renew.lola2.analysis;

import de.renew.lola2.LolaPlugin;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;

import java.util.ArrayList;
import java.util.Arrays;


/**
 * This is the main class for analyzing nets with LoLA. So far it offers the
 * capabilities to check if places are bounded and if transitions are dead.
 *
 * @author hewelt, wagner
 *
 */
public class LolaAnalyzer {
    private static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                    .getLogger(LolaAnalyzer.class);
    public static final int TIMEOUT = 5;

    /**
     * Points to the location, where the lola binaries reside
     */
    protected String lolaPath;

    public LolaAnalyzer() {
        lolaPath = LolaPlugin.getLolaPath();
    }

    public LolaAnalyzer(String lolaPath) {
        this.lolaPath = lolaPath;
    }

    /**
     * Calls the lola binaries for a given netFile, taskFile and lolaCommand.
     * Returns the output of the call (maybe an error) capsulated in a
     * LolaResult object
     *
     * @param netFile
     *            - The net in lola file format
     * @param taskFile
     *            - The file which contains the task, can be null
     * @param lolaCommand
     *            - The appropriate command for the given task
     *
     * @return a @see LolaResult object
     * @throws LolaResultException
     */
    public DirectLolaResult callLola(String formula, File netFile) {
        return callLola(formula, netFile, new String[] {});
    }

    public DirectLolaResult callLola(String formula, File netFile,
                                     String[] parameter) {
        Runtime myrun = Runtime.getRuntime();
        /*
         * construct lola call
         */
        File lolaBin = new File(lolaPath, LolaHelper.lolaCommand);
        if (logger.isDebugEnabled()) {
            logger.debug("LolaAnalyzer.callLola: lolaBin, netFile, formula, parameter are "
                            + lolaBin + " " + netFile + " " + formula + " "
                            + parameter);
        }

        // json file for output
        File jsonFile = null;
        try {
            jsonFile = File.createTempFile("lolaoutput", ".json",
                            LolaHelper.findTmpDir());
        } catch (IOException ioe) {
            logger.error("[Lola] Could not create temporary file for JSON output");
        }

        ArrayList<String> argList = new ArrayList<String>();
        argList.add(lolaBin.toString());
        argList.add(netFile.toString());
        argList.add("--timelimit=" + TIMEOUT);
        argList.add("--json=" + jsonFile.getAbsolutePath());
        if (formula != null) {
            argList.add("--formula=" + formula);
        }
        argList.addAll(Arrays.asList(parameter));
        String[] args = new String[argList.size()];
        args = argList.toArray(args);

        if (logger.isInfoEnabled()) {
            logger.info("[Lola] Executing " + renderCommand(args));
        }

        // lola puts output runtime information on standard error ("This output 
        // is purely informational and may change in future versions.") and json 
        // output on standard out.  
        StringBuilder outputBuilder = new StringBuilder();
        StringBuilder errorBuilder = new StringBuilder();

        int lolaRetVal = 1;
        try {
            Process lolproc = myrun.exec(args);
            lolaRetVal = lolproc.waitFor();

            BufferedReader outputReader = new BufferedReader(
                            new FileReader(jsonFile));

            //            BufferedReader outputReader = new BufferedReader(new InputStreamReader(lolproc.getInputStream()));
            BufferedReader errorReader = new BufferedReader(
                            new InputStreamReader(lolproc.getErrorStream()));

            String line = "";
            while ((line = outputReader.readLine()) != null) {
                outputBuilder.append(line);
                outputBuilder.append(System.lineSeparator());
                logger.debug("[callLola OUTPUT]" + line);
            }
            while ((line = errorReader.readLine()) != null) {
                errorBuilder.append(line);
                errorBuilder.append(System.lineSeparator());
                logger.debug("[callLola ERROR]" + line);
            }

            logger.debug("[Lola] Call returned exit value " + lolaRetVal);
            outputReader.close();
            errorReader.close();
        } catch (IOException e) {
            logger.error("[Lola] Execution failed");
            e.printStackTrace();
        } catch (InterruptedException e) {
            logger.error("[Lola] Execution interrupted");
            e.printStackTrace();
        }

        DirectLolaResult lolaResult = new DirectLolaResult(lolaRetVal,
                        outputBuilder.toString(), errorBuilder.toString());
        return lolaResult;
    }

    private String renderCommand(String[] args) {
        String cmd = "";
        for (int i = 0; i < args.length; i++) {
            String arg = args[i];
            if (arg.contains(" ")) {
                arg = "\"" + arg + "\"";
            }
            cmd += arg;
            if (i < args.length - 1) {
                cmd += " ";
            }
        }
        return cmd;
    }
}