package de.renew.gui.pnml.creator;

import org.w3c.dom.Element;

import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.framework.FigureEnumeration;

import de.renew.gui.ArcConnection;
import de.renew.gui.CPNDrawing;
import de.renew.gui.CPNTextFigure;
import de.renew.gui.DeclarationFigure;
import de.renew.gui.PlaceFigure;
import de.renew.gui.TransitionFigure;
import de.renew.gui.VirtualPlaceFigure;
import de.renew.gui.pnml.converter.Converter;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;


/**
 * Creator of a PNML {@code net} tag for a given {@link CPNDrawing}.
 * The PNML net type (stored in the {@code type} attribute at the created {@code net} tag)
 * is given in {@link #NetCreator(String)}.
 * The {@link CPNDrawing} is given to {@link #createElement(PNMLCreator, Object)}.
 */
public class NetCreator extends ElementCreator {
    public static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                    .getLogger(NetCreator.class);
    private List<PlaceFigure> placeList = new ArrayList<PlaceFigure>();

    /**
     * Creates a new {@link NetCreator} for the given PNML net type.
     * It will get stored in the {@code type} attribute at the created {@code net} tag.
     */
    public NetCreator(String netType, boolean toolInsc) {
        // Create a new Converter so its shadow net is not yet set.
        super("net", new Converter(netType, toolInsc));
    }

    private void connectNewStart(VirtualPlaceFigure virPlace,
                                 ArcConnection arc) {
        while (virPlace.getSemanticPlaceFigure() instanceof VirtualPlaceFigure) {
            virPlace = (VirtualPlaceFigure) virPlace.getSemanticPlaceFigure();
        }
        PlaceFigure figure = virPlace.getSemanticPlaceFigure();
        arc.disconnectStart();
        arc.connectStart(figure.connectorAt(figure.center()));
    }

    private void connectNewEnd(VirtualPlaceFigure virPlace, ArcConnection arc) {
        while (virPlace.getSemanticPlaceFigure() instanceof VirtualPlaceFigure) {
            virPlace = (VirtualPlaceFigure) virPlace.getSemanticPlaceFigure();
        }
        PlaceFigure figure = virPlace.getSemanticPlaceFigure();
        arc.disconnectEnd();
        arc.connectEnd(figure.connectorAt(figure.center()));
    }

    /**
     * Creates a PNML net tag containing all net elements
     * that is then returned as an {@link Element}.
     * The contained net elements are generated by their respective
     * {@link ElementCreator} implementations.
     * This method expects a {@link CPNDrawing} to have been given to
     * {@link ElementCreator#createElement(PNMLCreator, Object)}.
     */
    @Override
    protected Element doCreateElement(PNMLCreator pnmlCreator) {
        Element netEle = super.doCreateElement(pnmlCreator);
        CPNDrawing net = (CPNDrawing) getObject();
        Converter converter = getConverter();
        converter.setShadowNet(net);

        String netID = converter.getNetID();
        String netType = converter.getType();

        netEle.setAttribute("id", "net_" + netID);
        netEle.setAttribute("type", netType);

        // create name
        AttributeCreator name = new AttributeCreator(pnmlCreator, "name");
        netEle.appendChild(name.createElement(net.getName()));

        Element pageEle;

        // Do we want to create a page tag within the net tag?
        if (converter.getNetConverter().createPageTag()) {
            // Yes, create and append one.
            // All other tags get nested in this page tag.
            pageEle = pnmlCreator.createElement("page");
            pageEle.setAttribute("id", "page_" + netID);
            netEle.appendChild(pageEle);
        } else {
            // No, just put everything directly into the net tag.
            pageEle = netEle;
        }


        FigureEnumeration figures = net.figures();


        // Map from place IDs to the annotations of the virtual places that link it
        Hashtable<Integer, List<Figure>> virtualAnnotations = new Hashtable<Integer, List<Figure>>();

        // Fill the map by iterating over all figures...
        while (figures.hasMoreElements()) {
            Figure figure = figures.nextElement();

            // ...and if the figure is a virtual place,
            // add all of its annotations into the map for the original place's ID
            if (figure instanceof VirtualPlaceFigure) {
                VirtualPlaceFigure virPlace = (VirtualPlaceFigure) figure;
                VirtualPlaceCreator virCreator = new VirtualPlaceCreator(
                                converter);
                Element toolSpec = pnmlCreator.createToolspecific();
                toolSpec.appendChild(virCreator.createElement(pnmlCreator,
                                virPlace));
                pageEle.appendChild(toolSpec);
                // ID of original place
                Integer orgid = new Integer(
                                virPlace.getSemanticPlaceFigure().getID());


                // List of annotations
                List<Figure> annos;
                if (virtualAnnotations.containsKey(orgid)) {
                    annos = virtualAnnotations.get(orgid);
                } else {
                    annos = new ArrayList<Figure>();
                    virtualAnnotations.put(orgid, annos);
                }

                // Add this virtual place's annotations:
                FigureEnumeration figureenu = virPlace.children();
                while (figureenu.hasMoreElements()) {
                    Figure nextFig = figureenu.nextFigure();
                    annos.add(nextFig);
                }
            }
        }

        // Iterate over all figures again...
        figures = net.figures();
        while (figures.hasMoreElements()) {
            Figure figure = figures.nextElement();
            if (figure instanceof PlaceFigure
                            && !(figure instanceof VirtualPlaceFigure)) {
                PlaceCreator placeCreator = new PlaceCreator(converter);
                PlaceFigure place = (PlaceFigure) figure;
                placeList.add(place);
                Element pnmlPlace = placeCreator.createElement(pnmlCreator,
                                figure);


                // Get the the annotation of virtual places linking this place...
                List<Figure> virtAnnos = virtualAnnotations
                                .get(new Integer(place.getID()));

                // ...and add them to the PNML:
                if (virtAnnos != null) {
                    for (Figure fig : virtAnnos) {
                        pnmlPlace.appendChild(getNetConverter()
                                        .annotationToPNML((CPNTextFigure) fig,
                                                        pnmlCreator, true));
                    }
                }
                pageEle.appendChild(pnmlPlace);
            } else if (figure instanceof TransitionFigure) {
                TransitionCreator transitionCreator = new TransitionCreator(
                                converter);
                pageEle.appendChild(transitionCreator.createElement(pnmlCreator,
                                figure));
            } else if (figure instanceof ArcConnection) {
                ArcConnection arc = (ArcConnection) figure;
                Figure sourceFigure = arc.startFigure();
                Figure targetFigure = arc.endFigure();
                ArcCreator arcCreator = new ArcCreator(getConverter());
                if (sourceFigure instanceof VirtualPlaceFigure) {
                    VirtualPlaceFigure virPlace = (VirtualPlaceFigure) sourceFigure;
                    connectNewStart(virPlace, arc);
                    arcCreator.setVirStart(virPlace);
                }
                if (targetFigure instanceof VirtualPlaceFigure) {
                    VirtualPlaceFigure virPlace = (VirtualPlaceFigure) targetFigure;
                    connectNewEnd(virPlace, arc);
                    arcCreator.setVirEnd(virPlace);
                }
                pageEle.appendChild(
                                arcCreator.createElement(pnmlCreator, figure));
            } else if (figure instanceof DeclarationFigure) {
                pageEle.appendChild(converter.getNetConverter()
                                .annotationToPNML((DeclarationFigure) figure,
                                                pnmlCreator, false));
            } else if (figure instanceof CPNTextFigure) {
                if (((CPNTextFigure) figure).parent() == null) {
                    logger.error("NetCreator: Unknown type -> "
                                    + figure.getClass());
                }
            }
        }

        return netEle;
    }

    /**
     * A list of the visited place figures in the order the corresponding
     * PNML elements were created.
     */
    List<PlaceFigure> getPlaceList() {
        return placeList;
    }
}