package de.renew.gui;

import CH.ifa.draw.framework.ConnectionFigure;
import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.framework.FigureChangeAdapter;
import CH.ifa.draw.framework.FigureChangeEvent;
import CH.ifa.draw.framework.FigureEnumeration;
import CH.ifa.draw.framework.FilterContainer;

import CH.ifa.draw.io.SimpleFileFilter;

import CH.ifa.draw.standard.AbstractFigure;
import CH.ifa.draw.standard.CompositeFigure;
import CH.ifa.draw.standard.StandardDrawing;

import CH.ifa.draw.util.StorableInput;
import CH.ifa.draw.util.StorableOutput;

import de.renew.io.RNWFileFilter;

import de.renew.shadow.ShadowNet;
import de.renew.shadow.ShadowNetElement;
import de.renew.shadow.ShadowNetSystem;

import java.awt.Dimension;

import java.io.IOException;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.NoSuchElementException;


public class CPNDrawing extends StandardDrawing implements LayoutableDrawing {
    public static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                    .getLogger(CPNDrawing.class);
    private static FilterContainer filterContainer;

    /**
    * The shadow net of this drawing.
    * <p>
    * Transient, can be recalculated via
    * <code>buildShadow()</code>.
    * </p>
    */
    protected transient ShadowNet shadowNet = null;

    /**
     * The figure which should be displayed as representation
     * for instances of this net.
     * @serial
     */
    private AbstractFigure iconFigure = null;

    /**
     * Cache for all associations to
     * {@link FigureWithHighlight}s from their
     * highlight figures.
     * To point it out: the figure to be highlighted is
     * the key, and the figure with hilight is the value
     * of a pair in the hashtable.
     * <p>
     * Transient, will be rebuilt on deserialization.
     * </p>
     */
    private transient Hashtable<Figure, Figure> hilightMap = new Hashtable<Figure, Figure>();

    public CPNDrawing() {
        super();
    }

    @Override
    public void release() {
        super.release();
        discardShadow();
    }

    public ShadowNet getShadow() {
        return shadowNet;
    }

    static CPNDrawing findDrawing(Object errorObject) {
        // Determine the drawing containing the errorObject.
        if (errorObject instanceof ShadowNetElement) {
            return (CPNDrawing) ((ShadowNetElement) errorObject)
                            .getNet().context;
        }
        return null;
    }

    public void discardShadow() {
        if (shadowNet != null) {
            shadowNet.discard();
            shadowNet = null;
        }
    }

    /**
     * Calls the {@link ShadowHolder#buildShadow} method on all net
     * element figures of the given type.
     *
     * @param clazz  the <code>ShadowHolder</code> subclass to use
     *               as filter criterium
     */
    private void buildShadow(Class<?> clazz) {
        FigureEnumeration k = figures();

        while (k.hasMoreElements()) {
            Figure fig = k.nextFigure();

            if (fig instanceof ShadowHolder && clazz.isInstance(fig)) {
                ((ShadowHolder) fig).buildShadow(shadowNet);
            }
        }
    }

    public ShadowNet buildShadow(ShadowNetSystem netSystem) {
        discardShadow();
        shadowNet = new ShadowNet(getName(), netSystem);
        shadowNet.context = this;


        // Build shadows for declaration nodes (java-nets: only one per net!)
        buildShadow(DeclarationFigure.class);


        // Build shadows for nodes:
        buildShadow(NodeFigure.class);


        // Build shadows for connections:
        buildShadow(CH.ifa.draw.figures.LineConnection.class);


        // Build shadows for inscriptions:
        buildShadow(CPNTextFigure.class);

        return shadowNet;
    }

    public void setIconFigure(AbstractFigure iconFigure) {
        this.iconFigure = iconFigure;
    }

    public Figure getIconFigure() {
        return iconFigure;
    }

    /**
     * Removes a figure from the composite.
     * Additionally checks if the icon figure is removed.
     * Also checks if the highlight map has to be updated.
     */
    @Override
    public Figure remove(Figure figure) {
        if (figure == iconFigure) {
            iconFigure = null;
        }
        if (figure instanceof FigureWithHighlight) {
            Figure hilight = ((FigureWithHighlight) figure)
                            .getHighlightFigure();

            if (hilight != null) {
                hilightMap.remove(hilight);
            }
        }

        Figure result = super.remove(figure);

        if (figure instanceof CompositeFigure) {
            recomputeHilightMap();
        }
        return result;
    }

    /**
     * Writes the contained figures to the StorableOutput.
     */
    @Override
    public void write(StorableOutput dw) {
        super.write(dw);
        dw.writeStorable(iconFigure);
    }

    /**
     * Reads the contained figures from StorableInput.
     */
    @Override
    public void read(StorableInput dr) throws IOException {
        super.read(dr);
        if (dr.getVersion() >= 2) {
            try {
                iconFigure = (AbstractFigure) dr.readStorable();
            } catch (IOException e) {
                logger.error("Icon expected.");
                logger.debug("Icon expected.", e);
            }
            if (dr.getVersion() >= 3) {
                recomputeHilightMap();
            }
        }
    }

    @Override
    synchronized public void fillInGraph(GraphLayout layout) {
        FigureEnumeration k = figures();

        while (k.hasMoreElements()) {
            Figure f = k.nextFigure();

            if (f instanceof TransitionFigure || f instanceof PlaceFigure) {
                layout.addNode(f);

                // } else if (f instanceof ArcConnection) {
            } else if (f instanceof ConnectionFigure) {
                layout.addEdge((ConnectionFigure) f, 20);
            }
        }
    }

    void setHighlightFigure(final FigureWithHighlight node, final Figure fig) {
        Figure oldHighlight = node.getHighlightFigure();

        if (oldHighlight != null) {
            hilightMap.remove(oldHighlight);
        }
        node.setHighlightFigure(fig);
        if (fig != null) {
            hilightMap.put(fig, node);
            fig.addFigureChangeListener(new FigureChangeAdapter() {
                @Override
                public void figureRemoved(FigureChangeEvent e) {
                    setHighlightFigure(node, null);
                }
            });
            node.addFigureChangeListener(new FigureChangeAdapter() {
                @Override
                public void figureRemoved(FigureChangeEvent e) {
                    hilightMap.remove(fig);
                }
            });
        }
    }

    FigureWithHighlight getFigureForHighlight(Figure hilightFig) {
        try {
            return (FigureWithHighlight) hilightMap.get(hilightFig);
        } catch (NoSuchElementException e) {
            return null;
        }
    }

    @Override
    public Dimension defaultSize() {
        return new Dimension(535, 788);
    }

    /**
     * Deserialization method, behaves like default readObject
     * method except restoring default values for transient
     * fields.
     */
    private void readObject(java.io.ObjectInputStream in)
                    throws IOException, ClassNotFoundException {
        in.defaultReadObject();


        // The serialization mechanism constructs the object
        // with a less-than-no-arg-constructor, so that the
        // default values for transient fields have to be
        // reinitialized manually.
        hilightMap = new Hashtable<Figure, Figure>();


        // For full functionality we need to recompute some
        // tables.
        recomputeHilightMap();
    }

    /**
     * Recomputes the <code>hilightMap</code> by examining
     * all figures.
     */
    private void recomputeHilightMap() {
        hilightMap.clear();
        addToHilightMap(this);
    }

    /**
     * Adds the highlight figures of all figures contained in the
     * given <code>CompositeFigure</code> to the <code>hilightMap</code>.
     * Used by {@link #recomputeHilightMap}.
     *
     * @param container  CompositeFigure to scan for figures
     * with hilights.
     */
    private void addToHilightMap(CompositeFigure container) {
        FigureEnumeration figenumeration = container.figures();

        while (figenumeration.hasMoreElements()) {
            Figure fig = figenumeration.nextFigure();

            if (fig instanceof FigureWithHighlight) {
                Figure hilight = ((FigureWithHighlight) fig)
                                .getHighlightFigure();

                if (hilight != null) {
                    // logger.debug("Highlight for "+fig+" restored!");
                    hilightMap.put(hilight, fig);
                }
            }
            if (fig instanceof CompositeFigure) {
                addToHilightMap((CompositeFigure) fig);
            }
        }
    }

    @Override
    public String getWindowCategory() {
        return "Nets";
    }


    /**
     * Adds a figure to the list of figures
     * (as the superclass does).
     *
     * If the figure implements the interface<code>
     * CH.ifa.draw.framework.FigureWithID</code>, checks
     * its ID and assigns a new one if needed.
     */
    @Override
    public Figure add(Figure figure) {
        Figure result = super.add(figure);

        // If the figure can hilight other figures, put
        // its highlight figure into the map (if there
        // is any).
        if (figure instanceof FigureWithHighlight) {
            Figure hilight = ((FigureWithHighlight) figure)
                            .getHighlightFigure();

            if (hilight != null) {
                hilightMap.put(hilight, figure);
            }
        }


        // If a CompositeFigure is added, it may
        // contain a lot of other figures.
        if (figure instanceof CompositeFigure) {
            recomputeHilightMap();
        }

        return result;
    }

    //------------------------------------------------------------------------------   
    static public FilterContainer getFilterContainer() {
        if (filterContainer == null) {
            return new FilterContainer(new RNWFileFilter());
        } else {
            return filterContainer;
        }
    }

    @Override
    public SimpleFileFilter getDefaultFileFilter() {
        return getFilterContainer().getDefaultFileFilter();
    }

    @Override
    public HashSet<SimpleFileFilter> getImportFileFilters() {
        return getFilterContainer().getImportFileFilters();
    }

    @Override
    public HashSet<SimpleFileFilter> getExportFileFilters() {
        return getFilterContainer().getExportFileFilters();
    }

    /* (non-Javadoc)
     * @see CH.ifa.draw.framework.Drawing#getDefaultExtension()
     */
    @Override
    public String getDefaultExtension() {
        return getDefaultFileFilter().getExtension();
    }
}