/*
 * @(#)StandardDrawing.java 5.1
 *
 */
package CH.ifa.draw.standard;

import CH.ifa.draw.framework.Drawing;
import CH.ifa.draw.framework.DrawingChangeEvent;
import CH.ifa.draw.framework.DrawingChangeListener;
import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.framework.FigureChangeEvent;
import CH.ifa.draw.framework.FigureEnumeration;
import CH.ifa.draw.framework.FigureWithID;
import CH.ifa.draw.framework.FilterContainer;
import CH.ifa.draw.framework.Handle;

import CH.ifa.draw.io.IFAFileFilter;
import CH.ifa.draw.io.SimpleFileFilter;

import CH.ifa.draw.util.StorableInput;
import CH.ifa.draw.util.StorableOutput;

import de.renew.util.Lock;

import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Point;
import java.awt.Rectangle;

import java.io.File;
import java.io.IOException;
import java.io.ObjectInputStream;

import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Vector;


/**
 * The standard implementation of the Drawing interface.
 *
 * @see Drawing
 */
public class StandardDrawing extends CompositeFigure implements Drawing {
    public static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                    .getLogger(StandardDrawing.class);
    private static FilterContainer filterContainer;
    static String fgUntitled = "untitled";
    /*
     * Serialization support
     */
    private static final long serialVersionUID = -2602151437447962046L;

    /**
     * The registered listeners.
     * <p>
     * Transient, listeners must reregister themselves
     * after deserialization.
     * </p>
     */
    private transient Vector<DrawingChangeListener> fListeners;

    /**
     * Lock that blocks concurrent accesses to the drawing.
     * Unlike the previous solution, this lock counts the number of locks
     * and unlocks and will not prematurely regrant teh lock
     * in the case of multiple locks from the same thread.
     */
    private transient Lock fLock;

    /**
     * The name of the drawing, as it gets displayed and
     * can be used for references.
     * @serial
     **/
    private String fDrawingName = fgUntitled;

    /**
     * The named of the drawing, augmented by path and extension.
     * This information is used to save the drawing. Therefore
     * it doesn't make sense to keep it on serialization.
     **/
    private transient File fDrawingFilename = null;

    /**
     * Tells whether the drawing was modified since the
     * last save. Therefore it doesn't make sense to keep
     * the information on serialization.
     **/
    private transient boolean modified = false;

    /**
     * Tells whether a backup was created on the last save.
     * If <code>false</code>, a backup copy of an old file
     * with the same name will be made before the current
     * version gets written on the next save. Therefore
     * it doesn't make sense to keep the information on
     * serialization.
     **/
    private transient boolean fBackupStatus = false;
    @SuppressWarnings("unused")
    private int drawingSerializedDataVersion = 1;

    // ---- ID-Management -----------------------------------    
    
    /**
     * Caches the greatest used ID by any known figure.
     * Updated by <code>recomputeIDCache()</code>.
     * This field is transient because it is only a cache.
     */
    protected transient int maxUsedID = FigureWithID.NOID;
    /**
     * Caches used IDs in this and all subfigures.
     * Updated by <code>recomputeIDCache()</code>.
     * This field is transient because it is only a cache.
     */
    protected transient Hashtable<Integer, Figure> usedIDs = new Hashtable<Integer, Figure>();

    /**
     * Constructs the Drawing.
     */
    public StandardDrawing() {
        super();
        fListeners = new Vector<DrawingChangeListener>(2);
        fLock = new Lock();
    }

    @Override
    public String getName() {
        return fDrawingName;
    }

    @Override
    public void setName(String name) {
        fDrawingName = name;
    }

    public Dimension getSize() {
        return new Dimension(getBounds().width, getBounds().height);
    }

    @Override
    public File getFilename() {
        return fDrawingFilename;
    }

    @Override
    public void setFilename(File filename) {
        fDrawingFilename = filename;
    }

    /**
     * Return true, if a backup has been made since the
     * drawing has been generated.
     */
    @Override
    public boolean getBackupStatus() {
        return fBackupStatus;
    }

    /**
     * Inform the drawing that a backup has been generated.
     */
    @Override
    public void setBackupStatus(boolean status) {
        fBackupStatus = status;
    }

    /**
     * Adds a listener for this drawing.
     */
    @Override
    public void addDrawingChangeListener(DrawingChangeListener listener) {
        fListeners.addElement(listener);
    }

    /**
     * Removes a listener from this drawing.
     */
    @Override
    public void removeDrawingChangeListener(DrawingChangeListener listener) {
        fListeners.removeElement(listener);
    }

    /**
     * Adds a listener for this drawing.
     */
    @Override
    public Enumeration<DrawingChangeListener> drawingChangeListeners() {
        return fListeners.elements();
    }

    /**
     * Removes the figure from the drawing and releases it.
     * Also checks if the ID cache has to be updated.
     */
    @Override
    public Figure remove(Figure figure) {
        if (figure instanceof FigureWithID) {
            freeID((FigureWithID) figure);
        }

        // ensure that we remove the top level figure in a drawing
        Figure result = null;
        if (figure.listener() != null) {
            figure.listener().figureRequestRemove(
                            new FigureChangeEvent(figure, null));
            result = figure;
        }

        if (figure instanceof CompositeFigure) {
            recomputeIDCache();
        }
        return result;
    }

    /**
     * Handles a removeFromDrawing request that
     * is passed up the figure container hierarchy.
     * @see CH.ifa.draw.framework.FigureChangeListener
     */
    @Override
    public void figureRequestRemove(FigureChangeEvent e) {
        Figure figure = e.getFigure();
        if (fFigures.contains(figure)) {
            modified = true;
            fFigures.removeElement(figure);
            figure.removeFromContainer(this); // will invalidate figure
            figure.release();
        } else {
            logger.error("Attempt to remove non-existing figure");
        }
    }

    /**
     * Invalidates a rectangle and merges it with the
     * existing damaged area.
     * @see CH.ifa.draw.framework.FigureChangeListener
     */
    @Override
    public void figureInvalidated(FigureChangeEvent e) {
        modified = true;
        if (fListeners != null) {
            for (int i = 0; i < fListeners.size(); i++) {
                DrawingChangeListener l = fListeners.elementAt(i);
                l.drawingInvalidated(new DrawingChangeEvent(this,
                                e.getInvalidatedRectangle()));
            }
        }
    }

    /**
     * Forces an update
     */
    @Override
    public void figureRequestUpdate(FigureChangeEvent e) {
        if (fListeners != null) {
            for (int i = 0; i < fListeners.size(); i++) {
                DrawingChangeListener l = fListeners.elementAt(i);
                l.drawingRequestUpdate(new DrawingChangeEvent(this, null));
            }
        }
    }

    /**
     * Checks whether the drawing has some accumulated damage
     * and informs all views about the required update,
     * if neccessary.
     */
    @Override
    public synchronized void checkDamage() {
        final StandardDrawing object = this;
        EventQueue.invokeLater(new Runnable() {
            @Override
            public void run() {
                Enumeration<DrawingChangeListener> each = fListeners.elements();
                while (each.hasMoreElements()) {
                    DrawingChangeListener l = each.nextElement();
                    l.drawingRequestUpdate(
                                    new DrawingChangeEvent(object, null));
                }
            }
        });
    }

    /**
     * Return's the figure's handles. This is only used when a drawing
     * is nested inside another drawing.
     */
    @Override
    public Vector<Handle> handles() {
        Vector<Handle> handles = new Vector<Handle>();
        handles.addElement(new NullHandle(this, RelativeLocator.northWest()));
        handles.addElement(new NullHandle(this, RelativeLocator.northEast()));
        handles.addElement(new NullHandle(this, RelativeLocator.southWest()));
        handles.addElement(new NullHandle(this, RelativeLocator.southEast()));
        return handles;
    }

    /**
     * Gets the display box. This is the union of all figures.
     */
    @Override
    public Rectangle displayBox() {
        Rectangle box = null;
        FigureEnumeration k = figures();
        while (k.hasMoreElements()) {
            Figure f = k.nextFigure();
            if (f.isVisible()) {
                Rectangle r = f.displayBox();
                if (box == null) {
                    box = r;
                } else {
                    box.add(r);
                }
            }
        }
        if (box == null) {
            return new Rectangle(0, 0, 100, 100);
        } else {
            return new Rectangle(box.x - 10, box.y - 10, box.width + 20,
                            box.height + 20);
        }
    }

    @Override
    public void basicDisplayBox(Point p1, Point p2) {
    }

    /**
     * Acquires the drawing lock.
     */
    @Override
    public void lock() {
        fLock.lock();
    }

    /**
     * Releases the drawing lock.
     */
    @Override
    public void unlock() {
        fLock.unlock();
    }

    private void readObject(ObjectInputStream s)
                    throws ClassNotFoundException, IOException {
        s.defaultReadObject();

        fListeners = new Vector<DrawingChangeListener>(2);
        fLock = new Lock();
        

        // The serialization mechanism constructs the object
        // with a less-than-no-arg-constructor, so that the
        // default values for transient fields have to be
        // reinitialized manually.
        maxUsedID = FigureWithID.NOID;
        usedIDs = new Hashtable<Integer, Figure>();


        // For full functionality we need to recompute some
        // tables.
        recomputeIDCache();
    }

    /**
     * Returns whether drawing has been modified since last save.
     */
    @Override
    public boolean isModified() {
        return modified;
    }

    @Override
    public void clearModified() {
        modified = false;
    }

    @Override
    public Rectangle getBounds() {
        return displayBox();
    }

    @Override
    public Dimension defaultSize() {
        return new Dimension(430, 406);
    }

    /**
     * Writes the contained figures to the StorableOutput.
     */
    @Override
    public void write(StorableOutput dw) {
        super.write(dw);
    }

    /**
     * Reads the contained figures from StorableInput.
     */
    @Override
    public void read(StorableInput dr) throws IOException {
        super.read(dr);
        modified = false;
        recomputeIDCache();
    }

    @Override
    public String getWindowCategory() {
        return "JHotDrawings";
    }

    //------------------------------------------------------------------------------   
    static public FilterContainer getFilterContainer() {
        if (filterContainer == null) {
            return new FilterContainer(new IFAFileFilter());
        } else {
            return filterContainer;
        }
    }

    @Override
    public SimpleFileFilter getDefaultFileFilter() {
        return getFilterContainer().getDefaultFileFilter();
    }

    @Override
    public HashSet<SimpleFileFilter> getImportFileFilters() {
        return getFilterContainer().getImportFileFilters();
    }

    @Override
    public HashSet<SimpleFileFilter> getExportFileFilters() {
        return getFilterContainer().getExportFileFilters();
    }

    /* (non-Javadoc)
    * @see CH.ifa.draw.framework.Drawing#getDefaultExtension()
    */
    @Override
    public String getDefaultExtension() {
        return getDefaultFileFilter().getExtension();
    }

    @Override
    public void init() {
    }

    @Override
    public Drawing add(Drawing drawing) {
        FigureEnumeration figures = drawing.figures();
        while (figures.hasMoreElements()) {
            Figure figure = figures.nextElement();
            add(figure);
        }
        return drawing;
    }

    @Override
    public Drawing add(Drawing drawing, int x, int y) {
        FigureEnumeration figures = drawing.figures();
        while (figures.hasMoreElements()) {
            Figure figure = figures.nextElement();
            add(figure);
            figure.moveBy(x, y);
        }
        return drawing;
    }

    @Override
    public boolean isStorable() {
        return true;
    }

    /**
     * Adds a figure to the list of figures
     * (as the superclass does).
     *
     * If the figure implements the interface<code>
     * CH.ifa.draw.framework.FigureWithID</code>, checks
     * its ID and assigns a new one if needed.
     */
    @Override
    public Figure add(Figure figure) {
            // The ID check has to be done before the
            // figure is added to the list to avoid
            // collision of the figure with itself.
            // If figure is capable of holding an ID,
            // check its ID and assign a new one, if
            // needed.
            if (figure instanceof FigureWithID) {
                // If the cache is uninitialized,
                // recompute its value.
                if (maxUsedID == FigureWithID.NOID) {
                    recomputeIDCache();
                }
                checkAndAssignID((FigureWithID) figure);
    
                // Now the figure can be added to the list.
            }
            Figure result = super.add(figure);
    
            // If a CompositeFigure is added, it may
            // contain a lot of other figures.
            if (figure instanceof CompositeFigure) {
                recomputeIDCache();
            }
    
            return result;
        }

    /**
     * Checks the figure if it has already a
     * legal ID. If neccessary, generates new
     * unique ID and assigns it.
     */
    private void checkAndAssignID(FigureWithID figure) {
        int oldID = figure.getID();
        int newID = oldID;
        FigureWithID inCache = (FigureWithID) usedIDs.get(oldID);
    
        if ((inCache != null) && (inCache != figure)) {
            // The old ID is already used by another figure,
            // so reset it temporarily to NOID.
            newID = FigureWithID.NOID;
        }
        if (newID == FigureWithID.NOID) {
            newID = newUniqueID();
            figure.setID(newID);
        }
        usedIDs.put(newID, figure);
    }

    /**
     * Generates a new ID not used in the list of figures.
     * @see CH.ifa.draw.framework.FigureWithID
     */
    private int newUniqueID() {
        maxUsedID++;
    
        if (usedIDs.containsKey(maxUsedID)) {
            boolean resetOnce = false;
    
            while (usedIDs.containsKey(maxUsedID)) {
                maxUsedID++;
    
                if (maxUsedID == Integer.MIN_VALUE) {
                    maxUsedID = 1;
    
                    if (resetOnce) {
                        throw new RuntimeException(
                                        "Maximum numnber of figures exeeded.");
                    } else {
                        resetOnce = true;
                    }
                }
            }
        }
    
        return maxUsedID;
    }

    /**
     * Recomputes the ID cache (maxUsedID and usedIDs)
     * and eliminates IDs used more than once.
     */
    protected void recomputeIDCache() {
        // To ensure that no ID will be reassigned
        // even after it has been freed, do never
        // reset the greatest ID cache.
        // However, after closing the drawing, this
        // value will be reset anyway...
        // maxUsedID = FigureWithID.NOID;
        Vector<Figure> offendingFigures = new Vector<Figure>();
    
        usedIDs.clear();
    
        addToIDCache(this, offendingFigures);
    
        Enumeration<Figure> figureList = offendingFigures.elements();
    
        while (figureList.hasMoreElements()) {
            checkAndAssignID((FigureWithID) figureList.nextElement());
        }
    }

    /**
     * Do not call this method directly, use
     * <code>recomputeIDCache()</code> instead.
     * @param container CompositeFigure to scan for figures with ID.
     * @param offendingFigures Collects figures with illegal IDs.
     */
    private void addToIDCache(CompositeFigure container, Vector<Figure> offendingFigures) {
        FigureEnumeration knownFigures = container.figures();
        Figure figure;
        int usedID;
        FigureWithID inCache;
    
    
        // Iterate through all known Figures and update the
        // greatest seen ID.
        // Also update the Hashtable of used IDs and check if
        // some ID is already used by some other figure.
        // If there are CompositeFigures contained in the
        // drawing (like GroupFigures), recurse into them.
        while (knownFigures.hasMoreElements()) {
            figure = knownFigures.nextFigure();
            if (figure instanceof FigureWithID) {
                usedID = ((FigureWithID) figure).getID();
                if (usedID == FigureWithID.NOID) {
                    offendingFigures.addElement(figure);
                } else {
                    inCache = (FigureWithID) usedIDs.get(usedID);
                    if ((inCache == null) || (inCache == figure)) {
                        usedIDs.put(usedID, figure);
                        if (usedID > maxUsedID) {
                            maxUsedID = usedID;
                        }
                    } else {
                        // An ID is used twice.This will be silently corrected
                        // by the caller of this method.
                        // logger.debug("ID used more than once: "+usedID);
                        offendingFigures.addElement(figure);
                    }
                }
            } else if (figure instanceof CompositeFigure) {
                addToIDCache((CompositeFigure) figure, offendingFigures);
            }
        }
    }
    
    /**
     * Tries to assign a given id to the figure.
     */
    public void assignID(FigureWithID figure, int id) {
        // check if the new id can be given to the figure:        
        FigureWithID inCache = (FigureWithID) usedIDs.get(id);
        if ((inCache != null) && (inCache != figure)) {
            throw new IllegalArgumentException("The id is already in use!");
        }

        // remove old mapping from cache:
        usedIDs.remove(figure.getID());

        // set new id and add to cache:
        figure.setID(id);
        usedIDs.put(id, figure);
    }

    /**
     * Frees up the ID used by the given figure.
     */
    private void freeID(FigureWithID figure) {
        // Check if the ID was really cached
        // before freeing it up.
        Integer usedID = figure.getID();

        if (usedIDs.get(usedID) == figure) {
            usedIDs.remove(usedID);
        }
    }

    /**
     * Returns the FigureWithID currently assigned to
     * the given ID. If no figure is assigned to that
     * ID, returns <code>null</code>.
     * <p>
     * It is possible that the returned figure is not
     * contained in the <code>figures()</code> enumeration.
     * Then it was found in some CompositeFigure contained
     * in the enumeration.
     * </p>
     * @see CH.ifa.draw.framework.FigureWithID
     */
    public FigureWithID getFigureWithID(int id) {
        return (FigureWithID) usedIDs.get(id);
    }    
}