/*
 * @(#)ConnectionTool.java 5.1
 *
 */
package CH.ifa.draw.standard;

import CH.ifa.draw.framework.ConnectionFigure;
import CH.ifa.draw.framework.Connector;
import CH.ifa.draw.framework.Drawing;
import CH.ifa.draw.framework.DrawingEditor;
import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.framework.FigureEnumeration;

import CH.ifa.draw.util.Geom;

import java.awt.Point;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.List;


/**
 * A tool that can be used to connect figures, to split
 * connections, and to join two segments of a connection.
 * ConnectionTools turns the visibility of the Connectors
 * on when it enters a figure.
 * The connection object to be created is specified by a prototype.
 * <hr>
 * <b>Design Patterns</b><P>
 * <img src="images/red-ball-small.gif" width=6 height=6 alt=" o ">
 * <b><a href=../pattlets/sld029.htm>Prototype</a></b><br>
 * ConnectionTools creates the connection by cloning a prototype.
 * <hr>
 *
 * @see ConnectionFigure
 * @see Object#clone
 */
public class ConnectionTool extends UndoableTool {

    /**
     * the anchor point of the interaction
     */
    private Connector fStartConnector;
    private Connector fEndConnector;
    private Connector fConnectorTarget = null;
    private Figure fTarget = null;

    /**
     * the currently created figure
     */
    private ConnectionFigure fConnection = null;

    /**
     * the currently manipulated connection point
     */
    private int fSplitPoint;

    /**
     * the currently edited connection
     */
    private ConnectionFigure fEditedConnection = null;

    /**
     * the prototypical figure that is used to create new
     * connections.
     */
    private ConnectionFigure fPrototype;

    public ConnectionTool(DrawingEditor editor, ConnectionFigure prototype) {
        super(editor);
        fPrototype = prototype;
    }

    /**
     * Handles mouse move events in the drawing view.
     */
    @Override
    public void mouseMove(MouseEvent e, int x, int y) {
        trackConnectors(e, x, y);
    }

    /**
     * Manipulates connections in a context dependent way. If the
     * mouse down hits a figure start a new connection. If the mousedown
     * hits a connection split a segment or join two segments.
     */
    @Override
    public void mouseDown(MouseEvent e, int x, int y) {
        ConnectionFigure connection = findConnection(x, y, drawing());
        if (connection != null) {
            if (!connection.joinSegments(x, y)) {
                fSplitPoint = connection.splitSegment(x, y);
                fEditedConnection = connection;
            } else {
                fEditedConnection = null;
            }
            changesMade();
        } else {
            fConnection = createConnection();
            fTarget = findConnectionStart(x, y, drawing());
            //logger.debug("Target: "+fTarget);
            if (fTarget != null) {
                fStartConnector = findConnector(x, y, fTarget);
                //logger.debug("Startconnector: "+fStartConnector);
                if (fStartConnector != null) {
                    Point p = new Point(x, y);

                    //logger.debug("Connection: "+fConnection);
                    fConnection.startPoint(p.x, p.y);
                    fConnection.endPoint(p.x, p.y);
                    view().add(fConnection);
                    changesMade();
                } else {
                    fConnection = null;
                }
            } else {
                fConnection = null;
            }
        }
    }

    /**
     * Adjust the created connection or split segment.
     */
    @Override
    public void mouseDrag(MouseEvent e, int x, int y) {
        Point p = new Point(x, y);
        if (fConnection != null) {
            trackConnectors(e, x, y);
            if (fConnectorTarget != null) {
                p = Geom.center(fConnectorTarget.displayBox());
            }
            fConnection.endPoint(p.x, p.y);
        } else if (fEditedConnection != null) {
            Point pp = new Point(x, y);
            fEditedConnection.setPointAt(pp, fSplitPoint);
        }
    }

    /**
     * Connects the figures if the mouse is released over another
     * figure.
     */
    @Override
    public void mouseUp(MouseEvent e, int x, int y) {
        Figure c = null;
        if (fStartConnector != null) {
            c = findTarget(x, y, drawing());
        }

        if (c != null) {
            fEndConnector = findConnector(x, y, c);
            if (fEndConnector != null) {
                fConnection.connectStart(fStartConnector);
                fConnection.connectEnd(fEndConnector);
                fConnection.updateConnection();
            }
        } else if (fConnection != null) {
            view().remove(fConnection);
            noChangesMade();
        }

        fConnection = null;
        fStartConnector = fEndConnector = null;
        editor().toolDone();
    }

    /**
     * Deactivates the tool and clears any unfinished connections,
     * connectors or target figures.
     **/
    @Override
    public void deactivate() {
        super.deactivate();
        if (fConnection != null) {
            view().remove(fConnection);
            noChangesMade();
            fConnection = null;
            fStartConnector = fEndConnector = null;
        }
        if (fTarget != null) {
            fTarget.connectorVisibility(false);
        }
    }

    /**
     * Creates the ConnectionFigure. By default the figure prototype is
     * cloned.
     */
    protected ConnectionFigure createConnection() {
        return (ConnectionFigure) ((Figure) fPrototype).clone();
    }

    /**
     * Finds a connectable figure target.
     */
    protected Figure findSource(int x, int y, Drawing drawing) {
        List<Figure> possibleSources = findAllConnectableFigures(x, y, drawing);

        for (Figure source : possibleSources) {
            if (source != null && fConnection != null && source.canConnect()
                            && fConnection.canConnectStart(source)) {
                return source;
            }
        }
        return null;
    }

    /**
     * Finds a connectable figure target.
     */
    protected Figure findTarget(int x, int y, Drawing drawing) {
        List<Figure> possibleTargets = findAllConnectableFigures(x, y, drawing);
        Figure start = fStartConnector.owner();

        for (Figure target : possibleTargets) {
            if (target != null && fConnection != null && target.canConnect()
                            && !target.includes(start)
                            && fConnection.canConnect(start, target)) {
                return target;
            }
        }
        return null;
    }

    /**
     * Finds an existing connection figure.
     */
    protected ConnectionFigure findConnection(int x, int y, Drawing drawing) {
        FigureEnumeration k = drawing.figuresReverse();
        while (k.hasMoreElements()) {
            Figure figure = k.nextElement();
            figure = figure.findFigureInside(x, y);
            if (figure != null && (figure instanceof ConnectionFigure)) {
                return (ConnectionFigure) figure;
            }
        }
        return null;
    }

    /**
     * Gets the currently created figure
     */
    protected ConnectionFigure createdFigure() {
        return fConnection;
    }

    protected void trackConnectors(MouseEvent e, int x, int y) {
        if (fConnection == null) {
            //logger.debug("fConnection is null!!!");
            return;
        }

        Figure c = null;

        if (fStartConnector == null) {
            c = findSource(x, y, drawing());
        } else {
            c = findTarget(x, y, drawing());
        }

        // track the figure containing the mouse
        if (c != fTarget) {
            if (fTarget != null) {
                fTarget.connectorVisibility(false);
            }
            fTarget = c;
            if (fTarget != null) {
                fTarget.connectorVisibility(true);
            }
        }

        Connector cc = null;
        if (c != null) {
            cc = findConnector(x, y, c);
        }
        if (cc != fConnectorTarget) {
            fConnectorTarget = cc;
        }

        view().checkDamage();
    }

    private Connector findConnector(int x, int y, Figure f) {
        return f.connectorAt(x, y);
    }

    /**
     * Finds a connection start figure.
     */
    protected Figure findConnectionStart(int x, int y, Drawing drawing) {
        List<Figure> possibleTargets = findAllConnectableFigures(x, y, drawing);
        for (Figure target : possibleTargets) {
            if ((target != null) && target.canConnect()
                            && fConnection.canConnectStart(target)) {
                return target;
            }
        }
        return null;
    }

    private List<Figure> findAllConnectableFigures(int x, int y,
                                                   Drawing drawing) {
        List<Figure> connectableFigures = new ArrayList<>();
        FigureEnumeration k = drawing.figuresReverse();
        while (k.hasMoreElements()) {
            Figure figure = k.nextFigure();
            if ((fConnection == null || !figure.includes(fConnection))
                            && figure.canConnect()) {
                if (figure.containsPoint(x, y)) {
                    connectableFigures.add(figure);
                }
            }
        }
        return connectableFigures;
    }

    protected Connector getStartConnector() {
        return fStartConnector;
    }

    protected Connector getEndConnector() {
        return fEndConnector;
    }

    protected Connector getTarget() {
        return fConnectorTarget;
    }
}