package de.renew.engine.simulator;

import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import CH.ifa.draw.DrawPlugin;

import de.renew.formalism.ccpn.CCPNPlugin;

import de.renew.gui.CPNDrawing;
import de.renew.gui.PlaceFigure;
import de.renew.gui.pnml.creator.PNMLCreator;

import de.renew.io.exportFormats.PNMLExportFormat;

import de.renew.net.CurryPlaceInstance;
import de.renew.net.NetElementID;
import de.renew.net.NetInstance;
import de.renew.net.Place;

import de.renew.rnrg.elements.CurryNode;
import de.renew.rnrg.elements.Node;
import de.renew.rnrg.gui.GraphDrawing;

import de.renew.unify.Impossible;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.stream.FactoryConfigurationError;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stax.StAXResult;


public class CCPNProcess {
    public static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                    .getLogger(CCPNProcess.class);

    private class SAXHandler extends DefaultHandler {
        private final static String CCPN_OUTPUT = "ccpnOutput";
        private final static String NET_MARKING = "netMarking";
        private final static String NET_MARKING_STOPPED = "stopped";
        private final static String NET_MARKING_STOPPED_TRUE = "true";
        private final static String PLACE_STATE = "placeState";
        private final static String TOKEN = "token";
        private final static String TOKEN_COUNT = "count";
        private final static String ERROR = "error";
        private final static String LOAD_ERROR = "loadError";
        private final static String LOAD_WARNING = "loadWarning";
        private final static String REACH_GRAPH = "reachGraph";
        private final static String INITIAL_STATE = "initialState";
        private final static String ID = "id";
        private final static String REF = "ref";
        private final static String NAME = "name";
        private final static String TARGET = "target";
        private final static String NODE = "node";
        private final static String EDGE = "edge";
        private StringBuffer errorStringBuffer = null;
        private StringBuffer loadErrorStringBuffer = null;
        private StringBuffer loadWarningStringBuffer = null;
        private Stack<StringBuffer> debugOutput;
        private Iterator<Place> placeIterator;
        private CurryPlaceInstance currentPlace;
        private StringBuffer currentTokenStr = null;
        private int currentTokenCount;
        private CurryNode startNode = null;
        private List<String> initialStates = null;
        private Map<String, NodeEntry> nodes = null;
        private NodeEntry currentNode = null;
        private EdgeEntry currentEdge;

        final class EdgeEntry {
            public final String name;
            public String description = "";
            public final String target;

            public EdgeEntry(String name, String target) {
                this.name = name;
                this.target = target;
            }
        }

        final class NodeEntry {
            public final CurryNode node;
            public final List<EdgeEntry> edges = new ArrayList<EdgeEntry>();

            public NodeEntry(CurryNode node) {
                this.node = node;
            }
        }

        @Override
        public void startElement(String uri, String localName, String qName,
                                 Attributes attributes) {
            if (CCPNProcess.logger.isDebugEnabled()) {
                CCPNProcess.logger.debug(CCPNProcess.class.getSimpleName()
                                + ": CCPN " + (stopped ? "(stopped) " : "")
                                + "returned: <" + qName + ">");
            }

            if (TOKEN.equalsIgnoreCase(qName)) {
                currentTokenStr = new StringBuffer();
                currentTokenCount = Integer.parseUnsignedInt(
                                attributes.getValue("", TOKEN_COUNT));
            } else if (PLACE_STATE.equalsIgnoreCase(qName)) {
                NetInstance netInstance = currentNode == null ? simNetInstance
                                : currentNode.node.getRootNetInstance();
                currentPlace = (CurryPlaceInstance) netInstance
                                .getInstance(placeIterator.next());
                currentPlace.resetMarking();
            } else if (NET_MARKING.equalsIgnoreCase(qName)) {
                placeIterator = places.iterator();
                if (NET_MARKING_STOPPED_TRUE.equalsIgnoreCase(
                                attributes.getValue("", NET_MARKING_STOPPED))) {
                    // Stop the continues update if the simulation stopped.
                    continuousUpdate = false;
                }
            } else if (ERROR.equalsIgnoreCase(qName)) {
                errorStringBuffer = new StringBuffer();
            } else if (LOAD_ERROR.equalsIgnoreCase(qName)) {
                loadErrorStringBuffer = new StringBuffer();
            } else if (LOAD_WARNING.equalsIgnoreCase(qName)) {
                loadWarningStringBuffer = new StringBuffer();
            } else if (EDGE.equalsIgnoreCase(qName)) {
                currentEdge = new EdgeEntry(attributes.getValue("", NAME),
                                attributes.getValue("", TARGET));
            } else if (NODE.equalsIgnoreCase(qName)) {
                try {
                    currentNode = new NodeEntry(new CurryNode(
                                    simNetInstance.getNet().makeInstance()));
                } catch (Impossible e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
                nodes.put(attributes.getValue("", ID), currentNode);
            } else if (INITIAL_STATE.equalsIgnoreCase(qName)) {
                initialStates.add(attributes.getValue("", REF));
            } else if (REACH_GRAPH.equalsIgnoreCase(qName)) {
                try {
                    startNode = new CurryNode(
                                    simNetInstance.getNet().makeInstance());
                } catch (Impossible e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
                nodes = new HashMap<String, NodeEntry>();
                initialStates = new ArrayList<String>();
            } else if (CCPNProcess.logger.isDebugEnabled()
                            && !CCPN_OUTPUT.equalsIgnoreCase(qName)) {
                if (debugOutput == null) {
                    debugOutput = new Stack<StringBuffer>();
                }
                debugOutput.push(new StringBuffer());
            }
        }

        @Override
        public void endElement(String uri, String localName, String qName) {
            if (TOKEN.equalsIgnoreCase(qName)) {
                if (CCPNProcess.logger.isDebugEnabled()) {
                    CCPNProcess.logger.debug(CCPNProcess.class.getSimpleName()
                                    + ": " + currentTokenCount + " token"
                                    + (currentTokenCount > 1 ? "s" : "")
                                    + " in " + currentPlace + " : "
                                    + currentTokenStr);
                }
                currentPlace.addToken(currentTokenStr.toString(),
                                currentTokenCount);
                currentTokenStr = null;
            } else if (PLACE_STATE.equalsIgnoreCase(qName)) {
                currentPlace.notifyListeners();
                currentPlace = null;
            } else if (NET_MARKING.equalsIgnoreCase(qName)) {
                placeIterator = null;
                if (continuousUpdate) {
                    getMarking();
                }
            } else if (ERROR.equalsIgnoreCase(qName)) {
                if (errorStringBuffer.length() == 0) {
                    CCPNProcess.logger.error("CCPN returned error.");
                } else {
                    CCPNProcess.logger.error(
                                    "CCPN: " + errorStringBuffer.toString());
                }

                errorStringBuffer = null;
            } else if (LOAD_ERROR.equalsIgnoreCase(qName)) {
                CCPNProcess.logger.error("CCPN errors while loading:\n"
                                + loadErrorStringBuffer.toString());
                loadErrorStringBuffer = null;
            } else if (LOAD_WARNING.equalsIgnoreCase(qName)) {
                CCPNProcess.logger.warn("CCPN warnings while loading\n:"
                                + loadWarningStringBuffer.toString());
                loadWarningStringBuffer = null;
            } else if (EDGE.equalsIgnoreCase(qName)) {
                currentNode.edges.add(currentEdge);
                currentEdge = null;
            } else if (NODE.equalsIgnoreCase(qName)) {
                currentNode = null;
            } else if (REACH_GRAPH.equalsIgnoreCase(qName)) {
                List<Node> gNodes = new ArrayList<Node>(nodes.size() + 1);
                for (String is : initialStates) {
                    startNode.addEdge(nodes.get(is).node);
                }
                gNodes.add(startNode);
                for (NodeEntry ni : nodes.values()) {
                    for (EdgeEntry e : ni.edges) {
                        ni.node.addEdge(e.name, e.description,
                                        nodes.get(e.target).node);
                    }
                    gNodes.add(ni.node);
                }

                DrawPlugin.getGui().openDrawing(new GraphDrawing(gNodes,
                                startNode, GraphDrawing.INSC_NOTHING));

                startNode = null;
                nodes = null;
                initialStates = null;
            } else if (debugOutput != null && !debugOutput.empty()) {
                StringBuffer buf = debugOutput.pop();
                CCPNProcess.logger.debug(CCPNProcess.class.getSimpleName()
                                + ": ignoring unknown element " + qName
                                + (buf.length() == 0 ? ""
                                                : " with contents: " + buf
                                                                .toString()));
            }
        }

        @Override
        public void characters(char[] ch, int start, int length) {
            if (currentTokenStr != null) {
                currentTokenStr.append(ch, start, length);
            } else if (errorStringBuffer != null) {
                errorStringBuffer.append(ch, start, length);
            } else if (loadErrorStringBuffer != null) {
                loadErrorStringBuffer.append(ch, start, length);
            } else if (loadWarningStringBuffer != null) {
                loadWarningStringBuffer.append(ch, start, length);
            } else if (currentEdge != null) {
                String str = String.valueOf(ch, start, length);
                currentEdge.description += str;
            } else if (debugOutput != null && !debugOutput.empty()) {
                debugOutput.peek().append(ch, start, length);
            }
        }

        @Override
        public void endDocument() {
            if (CCPNProcess.logger.isDebugEnabled()) {
                CCPNProcess.logger.debug(CCPNProcess.class.getSimpleName()
                                + ": CCPN output ended.");
            }
        }
    }

    final private Process process;
    final private List<Place> places;
    final private NetInstance simNetInstance;
    boolean continuousUpdate = false;
    boolean stopped = false;
    final XMLStreamWriter writer;

    static public CCPNProcess startProcess(final NetInstance netInstance,
                                           final CPNDrawing drawing) {
        if (logger.isDebugEnabled()) {
            logger.debug(CCPNProcess.class.getSimpleName()
                            + ": starting process...");
        }

        /** Prepare PNML **/
        PNMLCreator creator = new PNMLCreator(PNMLExportFormat.ptNetType, true);
        final List<PlaceFigure> placeFigs;
        try {
            placeFigs = creator.setDrawings(drawing);
        } catch (ParserConfigurationException e) {
            throw new RuntimeException(e);
        }

        List<Place> places = new ArrayList<Place>(placeFigs.size());

        for (PlaceFigure p : placeFigs) {
            places.add(netInstance.getNet()
                            .getPlaceWithID(new NetElementID(p.getID())));
        }

        /** Start process **/
        ProcessBuilder pb = new ProcessBuilder(CCPNPlugin.getCCPNLocation());

        CCPNProcess retProcess;
        try {
            retProcess = new CCPNProcess(pb.start(), places, netInstance);
        } catch (IOException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }

        /** Load net **/
        Document document;
        try {
            document = DocumentBuilderFactory.newInstance().newDocumentBuilder()
                            .newDocument();
        } catch (ParserConfigurationException e) {
            throw new RuntimeException(e);
        }
        Element loadElem = document.createElement("load");
        File filename = drawing.getFilename();
        if (filename != null) {
            try {
                loadElem.setAttribute("baseFile", filename.getCanonicalPath());
            } catch (DOMException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        loadElem.appendChild(document.adoptNode(creator.getDocumentElement()));
        retProcess.sendElement(loadElem);

        /** Read output **/
        retProcess.execute();
        return retProcess;
    }

    private void execute() {
        SimulationThreadPool.getCurrent().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    SAXParserFactory.newInstance().newSAXParser()
                                    .parse(new BufferedInputStream(
                                                    process.getInputStream()),
                                                    new SAXHandler());
                } catch (SAXException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                } catch (IOException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                } catch (ParserConfigurationException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
            }
        });
    }

    public CCPNProcess(Process process, List<Place> places,
                       NetInstance netInstance) {
        this.process = process;
        this.places = places;
        this.simNetInstance = netInstance;

        try {
            writer = XMLOutputFactory.newFactory().createXMLStreamWriter(
                            new BufferedOutputStream(process.getOutputStream()),
                            "UTF-8");
            writer.writeStartDocument();
        } catch (XMLStreamException e) {
            throw new RuntimeException(e);
        } catch (FactoryConfigurationError e) {
            throw new RuntimeException(e);
        }
    }

    public void terminate() {
        stopped = true;
        continuousUpdate = false;
        if (logger.isDebugEnabled()) {
            logger.debug(CCPNProcess.class.getSimpleName()
                            + ": closing XML stream to process.");
        }
        try {
            writer.writeEndDocument();
            writer.flush();
            writer.close();
        } catch (XMLStreamException e) {
            // ignore
        }
        try {
            process.getOutputStream().close();
        } catch (IOException e) {
            // ignore
        }
    }

    void run() {
        sendElement("run");
        getMarking();
    }

    void stop() {
        sendElement("stop");
        getMarking();
    }

    void step() {
        sendElement("step");
        getMarking();
    }

    void reachGraph() {
        sendElement("reachGraph");
    }

    private void getMarking() {
        sendElement("getMarking");
    }

    private void sendElement(Element elem) {
        if (stopped) {
            return;
        }

        if (CCPNProcess.logger.isDebugEnabled()) {
            CCPNProcess.logger.debug(CCPNProcess.class.getSimpleName()
                            + ": sending " + elem.getTagName()
                            + " DOM element");
        }

        try {
            // TODO if the transformer is reused, it mustn't be used multiple times concurrently
            // Uses OmitDeclXMLStreamWriter, because this does not work:
            // transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
            TransformerFactory.newInstance().newTransformer()
                            .transform(new DOMSource(elem), new StAXResult(
                                            new XMLOmitDeclStreamWriter(
                                                            writer)));
            writer.writeEndDocument();
            writer.flush();
        } catch (TransformerException e) {
            throw new RuntimeException(e);
        } catch (XMLStreamException e) {
            handleXMLStreamException(e);
        }
    }

    private void sendElement(String tag) {
        if (stopped) {
            return;
        }

        if (CCPNProcess.logger.isDebugEnabled()) {
            CCPNProcess.logger.debug(CCPNProcess.class.getSimpleName()
                            + ": sending " + tag + " empty element");
        }

        try {
            writer.writeEmptyElement(tag);
            // This is necessary so the entire empty element is written out:
            writer.writeEndDocument();
            writer.flush();
        } catch (XMLStreamException e) {
            handleXMLStreamException(e);
        }
    }

    private static void handleXMLStreamException(XMLStreamException e) {
        Throwable cause = e.getCause();
        if (cause instanceof IOException
                        && "Stream closed".equals(cause.getMessage())) {
            // ignore
            return;
        }

        throw new RuntimeException(e);
    }
}
