/*
 * @(#)StringUtil.java
 *
 */
package de.renew.util;

import java.io.File;

import java.net.URI;
import java.net.URISyntaxException;

import java.util.Vector;


/**
 * A utility class for advanced string handling, e.g. for filenames, paths,
 * extensions, replacing substring etc.
 * We use the following naming conventions:
 * <ul>
 * <i>An <b>absolute pathname</b> is a string which fully describes a
 * file (including path and extension), starting from a system-dependent
 * root string.
 * <i>A <b>relative pathname</b> is a pathname which is specified
 * relatively to another path and thus does not start with a
 * system-dependent root string.
 * <i>An <b>path</b> is a string which fully describes a directory.
 * We also use "relative" and "absolute" here.
 * <i>A <b>file name</b> is a pathname without the path and without
 * the extension.
 * <i>An <b>extended file name</b> is a pathname without the path
 * (but including the extension).
 * <i>An <b>extension</b> is a everything following the "dot" in
 * an extended file name. If there is not dot in an extended file name,
 * its extension is the empty string.
 * </ul>
 */
public class StringUtil {
    public static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                                                       .getLogger(StringUtil.class);

    /** The length of File.separator, which is usually 1. */
    public final static int separatorLength = File.separator.length();

    /** Returns the string resulting from replacing all appearences of
     *  <tt>what</tt> in <tt>str</tt> by <tt>by</tt>.
     *  @param str The original string.
     *  @param what The substring to look for.
     *  @param by The string by which <tt>what</tt> is replaced.
     *  @return The result of replacing all apprearences of <tt>what</tt>
     *          in <tt>str</tt> by <tt>by</tt>.
     */
    public static String replace(String str, String what, String by) {
        StringBuffer rpl = new StringBuffer();
        int whatlen = what.length();
        int index = 0;
        int lastIndex;
        while (true) {
            // search for next
            lastIndex = index;
            index = str.indexOf(what, lastIndex);
            if (index < 0) {
                break;
            }
            rpl.append(str.substring(lastIndex, index));
            rpl.append(by);
            index += whatlen;
        }
        rpl.append(str.substring(lastIndex));
        return rpl.toString();
    }

    /** Splits a string into parts defined by a seperation string.
     *  The seperation string itself is not put into the array
     *  elements. Empty elements are left out.
     *  In contrast to java's StringTokenizer, the seperation string
     *  is interpreted as a sequence of characters, not as a choice
     *  of different seperation characters.
     */
    public static String[] split(String str, final String sep) {
        Vector<String> entries = new Vector<String>();
        final int seplen = sep.length();
        int index = 0;
        int lastIndex;
        str = str + sep;
        while (true) {
            // search for next
            lastIndex = index;
            index = str.indexOf(sep, lastIndex);
            if (index < 0) {
                break;
            }
            if (index > lastIndex) {
                entries.addElement(str.substring(lastIndex, index));
            }
            index += seplen;
        }

        String[] result = new String[entries.size()];
        entries.copyInto(result);
        return result;
    }

    /** Converts only the first character of a String to
     *  upper case. The rest of the String or the empty
     *  String is not modified.
     */
    public static String firstToUpperCase(String str) {
        if (str.length() == 0) {
            return str;
        }
        return str.substring(0, 1).toUpperCase() + str.substring(1);
    }

    /** Converts only the first character of a String to
     *  lower case. The rest of the String or the empty
     *  String is not modified.
     */
    public static String firstToLowerCase(String str) {
        if (str.length() == 0) {
            return str;
        }
        return str.substring(0, 1).toLowerCase() + str.substring(1);
    }

    /** Returns whether the nth character in String str is in upper case.
     *  Returns false if n is out of bounds.
     */
    public static boolean isUpperCaseAt(String str, int n) {
        return n < str.length() && Character.isUpperCase(str.charAt(n));
    }

    /** Returns whether the nth character in String str is in lower case.
     *  Returns false if n is out of bounds.
     */
    public static boolean isLowerCaseAt(String str, int n) {
        return n < str.length() && Character.isLowerCase(str.charAt(n));
    }

    /** Counts the occurences of a substring within a String.
     *  Overlapping occurences are not counted
     *  (e.g. countOccurences("AAAA","AA")==2, not 3).
     */
    public static int countOccurences(String str, String sub) {
        int count = 0;
        int index = 0;
        int sublen = sub.length();
        while (true) {
            index = str.indexOf(sub, index);
            if (index < 0) {
                break;
            }
            index += sublen;
            ++count;
        }
        return count;
    }

    /** Counts the number of lines within a String. A String has one more line
     *  than the number of new-line characters it contains.
     */
    public static int countLines(String str) {
        return countOccurences(str, "\n") + 1;
    }

    /****************************************/

    /*                                      */
    /* File-related string editing methods  */
    /*                                      */


    /****************************************/
    private static int getLastSeparatorPos(String pathname) {
        int pos = pathname.lastIndexOf('/');
        if (pos == -1) {
            pos = pathname.lastIndexOf(File.separator);
        }
        return pos;
    }

    private static int getLastDotPos(String pathname) {
        int lsp = getLastSeparatorPos(pathname);
        int ldp = pathname.lastIndexOf(".");
        if (ldp > lsp) {
            return ldp;
        } else {
            return pathname.length();
        }
    }

    /** Removes the file extension from the filename.
     *  Also remove the dot separating the extension.
     */
    public static String stripFilenameExtension(String pathname) {
        return pathname.substring(0, getLastDotPos(pathname));
    }

    /** Returns the corresponding extended filename for a given
     *  pathname.
     */
    public static String getExtendedFilename(String pathname) {
        return pathname.substring(getLastSeparatorPos(pathname) + 1);
    }

    /** Returns the corresponding filename for a given
     *  pathname.
     */
    public static String getFilename(String pathname) {
        return pathname.substring(getLastSeparatorPos(pathname) + 1,
                                  getLastDotPos(pathname));
    }

    /** Returns the corresponding extension for a given
     *  pathname.
     */
    public static String getExtension(String pathname) {
        int firstAfterDot = getLastDotPos(pathname) + 1;
        if (firstAfterDot < pathname.length()) {
            return pathname.substring(firstAfterDot).toLowerCase();
        } else {
            return "";
        }
    }

    public static String getPath(String pathname) {
        int lastPos = getLastSeparatorPos(pathname);
        if (lastPos < 0) {
            return "";
        } else {
            return pathname.substring(0, lastPos);
        }
    }

    public static String makeRelative(String basePathname, String pathname) {
        // TO-DO: case that path is not a subpath of basePathname.
        if (pathname != null) {
            if (basePathname == null) {
                pathname = convertToSlashes(pathname);
            } else {
                // add seperator at end of basePathname:
                basePathname += File.separator;
                // find longest matching path-prefix:
                int index = 0;

                // find longest matching path-prefix:
                int size;
                while (true) {
                    // search for next
                    size = index;
                    index = basePathname.indexOf(File.separator, index)
                            + separatorLength;
                    if (index == -1 + separatorLength
                                || !pathname.startsWith(basePathname.substring(0,
                                                                                       index))) {
                        break;
                    }
                }
                String dotdots = "";
                if (size > 0) {
                    // size==0 would mean that either "base" was not
                    // absolute or it was a DOS path on a different drive. There
                    // is no way to make this path relative, so return the
                    // absolute pathname.
                    index = size;
                    while ((index = basePathname.indexOf(File.separator, index)) >= 0) {
                        index += separatorLength;
                        dotdots += "../";
                    }
                }
                pathname = dotdots + convertToSlashes(pathname.substring(size));
            }
        }
        return pathname;
    }

    public static String makeCanonical(String pathOrFilename) {
        String pathname = null;
        try {
            pathname = new File(pathOrFilename).getCanonicalPath();
            // Check for OS/2 network filename bug:
            if (pathname.equals(pathname.toUpperCase())) {
                // Only use the path information of the new String
                return getPath(pathname) + File.separator
                       + getExtendedFilename(pathOrFilename);
            }
            return pathname;
        } catch (java.io.IOException e) {
            // File does not exist, return the pathOrFilename itself.
            // Maybe try to make relative paths absolute "by hand"?
            return pathOrFilename;
        }
    }

    public static String makeAbsolute(String basePathname, String path) {
        if (basePathname == null || "".equals(basePathname)
                    || path.startsWith("/")
                    || (path.length() > 2 && path.charAt(1) == ':')) {
            // The base pathname is empty or the path was already absolute.
            return convertToSystem(path);
        } else {
            while (path.startsWith("../")) {
                path = path.substring(3);
                basePathname = getPath(basePathname);
            }
            return basePathname + File.separator + convertToSystem(path);
        }
    }

    /** Returns the result of replacing all system-dependent file
     *  separators by slashes.
     */
    public static String convertToSlashes(String path) {
        return replace(path, File.separator, "/");
    }

    /** Returns the result of replacing all slashes by
     *  system-dependent file separators.
     */
    public static String convertToSystem(String path) {
        return replace(path, "/", File.separator);
    }

    public static String[] splitPaths(String paths) {
        return split(paths, File.pathSeparator);
    }

    /** Returns a relative pathname for a given full class name.
     */
    public static String classToFile(String classname) {
        return replace(classname, ".", File.separator) + ".class";
    }

    /**
     * Strips off all leading and trailing white space
     * and substitutes sequences of white space by a single space.
     */
    public static String unspace(String arg) {
        StringBuffer result = new StringBuffer();
        int n = arg.length();
        boolean space = false;
        boolean first = true;
        for (int i = 0; i < n; i++) {
            char c = arg.charAt(i);
            if (Character.isWhitespace(c)) {
                space = true;
            } else {
                if (space && !first) {
                    result.append(' ');
                }
                space = false;
                first = false;
                result.append(c);
            }
        }
        return result.toString();
    }

    /**
     * Determine whether a string constitutes a valid
     * Java identifier or is empty.
     */
    public static boolean isNameOrEmpty(String name, boolean dotAllowed) {
        int i = 0;

        // We accept the empty string.
        while (i < name.length()) {
            // One letter.
            if (!Character.isJavaIdentifierStart(name.charAt(i++))) {
                return false;
            }

            // An arbitrary number of letters of digits.
            while (i < name.length()
                           && Character.isJavaIdentifierPart(name.charAt(i))) {
                i++;
            }

            // Either the end of the string or ...
            if (i < name.length()) {
                if (!dotAllowed) {
                    return false;
                }

                // ... a dot ...
                if (name.charAt(i++) != '.') {
                    return false;
                }

                // ... followed by something.
                if (i == name.length()) {
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * Relativizes one URI against another by looking at the
     * parent of the given base URI. The parent is taken into
     * consideration only if direct URI relativization fails.
     * <p>
     * This operation is a refinement of {@link URI#relativize}
     * that acts like the
     * {@link #makeRelative(String,String) makeRelative}
     * operation for filename strings.
     * If <code>baseURI.relativize(uri)</code> would return a
     * non-absolute URI, this method returns the same result.
     * </p>
     * <p>
     * The following condition holds always:
     * <code>baseURI.resolve(makeRelative(baseURI, uri)).equals(uri)</code>
     *
     * @param baseURI the base URI
     * @param uri the
     * @return an <code>URI</code> value
     **/
    public static URI makeRelative(URI baseURI, URI uri) {
        // First try: ask the URI itself
        URI result = baseURI.relativize(uri);


        // If the result was not relative, make a second try
        // with the parent path of the baseURI. This attempt
        // makes sense only if there is hope, therefore check
        // whether scheme and authority of both URIs match.
        if (result.isAbsolute()
                    && equalOrBothNull(baseURI.getScheme(), uri.getScheme())
                    && equalOrBothNull(baseURI.getAuthority(),
                                               uri.getAuthority())) {
            URI lastParent = null;
            int backsteps = 0;
            URI parent = getParent(baseURI, false);
            result = parent.relativize(uri);


            // if the result is still not relative, iterate
            // backwards through the base URI's path.
            while (result.isAbsolute() && !parent.equals(lastParent)) {
                lastParent = parent;
                backsteps++;
                parent = getParent(lastParent, true);
                result = parent.relativize(uri);
            }


            // if we were successful, add a "../" to the path
            // for each backstep.
            if (!result.isAbsolute()) {
                StringBuffer newPath = new StringBuffer();
                for (int i = 0; i < backsteps; i++) {
                    newPath.append("../");
                }
                newPath.append(result.getPath());
                try {
                    result = new URI(result.getScheme(), result.getAuthority(),
                                     newPath.toString(), result.getQuery(),
                                     result.getFragment());
                } catch (URISyntaxException e) {
                    logger.error(e.getMessage(), e);
                    result = uri;
                }
            }
        }
        assert baseURI.resolve(result).equals(uri) : "StringUtil.makeRelative result does not hold its contract."
        + "\n  given uri=" + uri + "\n  verified =" + baseURI.resolve(result)
        + "\n  result   =" + result + "\n  baseURI  =" + baseURI;
        return result;
    }

    private static boolean equalOrBothNull(Object one, Object two) {
        return (one == null && two == null) || (one != null && one.equals(two));
    }

    /**
     * Computes a parent URI from the given URI. This means that
     * <i>scheme</i> and <i>authority</i> remain unchanged, the
     * last segment of the <i>path</i> is removed, and both
     * <i>query</i> and <i>fragment</i> are dropped.
     * If the path of the given URI ends with a slash (meaning
     * it denotes a directory), the path will not be shortened
     * unless <code>directoriesToo</code> is set.
     * If the path denotes the root directory (slash only), it
     * will never be shortened.
     * <p>
     * This operation is somewhat like the {@link #getPath}
     * operation for filename strings.
     * </p>
     * @param uri the base URI to compute the parent from
     * @param directoriesToo whether URIs denoting directories
     *                       should also be shortened. If
     *                       <code>false</code>, a directory is
     *                       considered being its own parent.
     * @return the shortened URI.
     */
    public static URI getParent(URI uri, boolean directoriesToo) {
        String path = uri.getPath();
        String parent = null;
        if (path != null) {
            if (directoriesToo && path.endsWith("/") && (path.length() > 1)) {
                path = path.substring(0, path.length() - 1);
            }
            int lastPos = path.lastIndexOf("/");
            if (lastPos >= 0) {
                parent = path.substring(0, lastPos + 1);
            }
        }
        try {
            URI result = new URI(uri.getScheme(), uri.getAuthority(), parent,
                                 null, null);
            return result;
        } catch (URISyntaxException e) {
            logger.error(e.getMessage(), e);
            throw new IllegalArgumentException("StringUtil: Could not derive parent URI from "
                                               + uri);
        }
    }
}