package de.renew.expression;

import de.renew.unify.CalculationChecker;
import de.renew.unify.Impossible;
import de.renew.unify.StateRecorder;
import de.renew.unify.Variable;


/**
 * Upon evaluation I will spawn the evaluation of several
 * supexpressions. I will then compose a aggregate from the resulting values.
 *
 * @author Olaf Kummer
 **/
public abstract class AggregateExpression implements Expression {
    static final long serialVersionUID = -4094960769944117557L;

    /**
     * An array with the expressions that I must evaluate.
     **/
    private Expression[] expressions;

    /**
     * I am invertible if I have at least one invertible
     * subexpression. By default, I am supposed to be
     * invertible.
     **/
    private boolean invertible = true;

    /**
     * I (a aggregate expression) am created.
     *
     * @param expressions
     *   a aggregate that holds the expressions that must be evaluated
     **/
    public AggregateExpression(Expression[] expressions) {
        this.expressions = expressions;
        invertible = false;
        for (int i = 0; i < expressions.length; i++) {
            invertible = invertible || expressions[i].isInvertible();
        }
    }

    /**
     * Am I invertible? I look it up in the cached field.
     *
     * @return true, if at least one of my subexpressions is invertible.
     **/
    public boolean isInvertible() {
        return invertible;
    }

    public Expression[] getExpressions() {
        return expressions;
    }

    /**
     * My children determine my type.
     **/
    public abstract Class<?> getType();

    /**
     * This is a convenience method that prepares an array
     * that holds the result of the evaluation of the subexpressions.
     *
     * @param mapper
     *   a variable mapper which I must pass to the subexpressions
     * @param recorder
     *   a state recorder which I must pass to the subexpressions
     * @param checker
     *   a calculation checker which I must pass to the subexpressions
     * @param forEvaluation
     *   true, if we are trying an evaluation
     * @return the constructed aggregate, possibly incomplete
     **/
    protected Object[] makeArray(VariableMapper mapper, StateRecorder recorder,
                                 CalculationChecker checker,
                                 boolean forEvaluation)
            throws Impossible {
        int n = expressions.length;


        // We must store the results in variables first, because they
        // may be unknowns that are replaced by other objects later on.
        Variable[] vars = new Variable[n];
        for (int i = 0; i < n; i++) {
            Expression expr = expressions[i];
            if (forEvaluation) {
                vars[i] = new Variable(expr.startEvaluation(mapper, recorder,
                                                            checker), recorder);
            } else {
                vars[i] = new Variable(expr.registerCalculation(mapper,
                                                                recorder,
                                                                checker),
                                       recorder);
            }
        }


        // We can now copy the variable values. It is only safe to use this
        // array until the next unification that assigns values to
        // previously established unknowns.
        Object[] result = new Object[n];
        for (int i = 0; i < n; i++) {
            result[i] = vars[i].getValue();
        }

        return result;
    }

    /**
     * Create an aggregate. My subexpresssion will override this
     * factory method to create an aggregate of their choice using
     * the arguments that were generated by the subexpressions.
     *
     * @param args
     *   the objects to be referenced by the constructed aggregate
     * @param recorder
     *   a state recorder
     * @return the constructed aggregate, possibly incomplete
     **/
    protected abstract Object makeResultAggregate(Object[] args,
                                                  StateRecorder recorder)
            throws Impossible;

    /**
     * I will evaluate the subexpressions and put the results
     * into a aggregate.
     *
     * @param mapper
     *   a variable mapper which I must pass to the subexpressions
     * @param recorder
     *   a state recorder which I must pass to the subexpressions
     * @param checker
     *   a calculation checker which I must pass to the subexpressions
     * @return the constructed aggregate, possibly incomplete
     **/
    public Object startEvaluation(VariableMapper mapper,
                                  StateRecorder recorder,
                                  CalculationChecker checker)
            throws Impossible {
        return makeResultAggregate(makeArray(mapper, recorder, checker, true),
                                   recorder);
    }

    /**
     * I will register the subexpressions and put the results
     * into a aggregate.
     *
     * @param mapper
     *   a variable mapper which I must pass to the subexpressions
     * @param recorder
     *   a state recorder which I must pass to the subexpressions
     * @param checker
     *   a calculation checker which I must pass to the subexpressions
     * @return the constructed aggregate, possibly incomplete
     **/
    public Object registerCalculation(VariableMapper mapper,
                                      StateRecorder recorder,
                                      CalculationChecker checker)
            throws Impossible {
        return makeResultAggregate(makeArray(mapper, recorder, checker, false),
                                   recorder);
    }

    public String toString() {
        String result = "AggregateExpr("
                        + de.renew.util.Types.typeToString(getType()) + ": ";
        for (int i = 0; i < expressions.length; i++) {
            if (i > 0) {
                result = result + ", ";
            }
            result = result + expressions[i];
        }
        result = result + ")";
        return result;
    }
}