package de.renew.engine.simulator;

import de.renew.util.Semaphor;

import java.util.concurrent.Callable;
import java.util.concurrent.FutureTask;


/**
 * This class serves as a way to run a Callables of type \<T\> in a simulation thread and wait for its call() method to finish
 * It will notify the thread holding the Semaphor after execution has been finished.
 * @author Benjamin Schleinzer
 *
 * @param <T> the type of the passed Callable
 */
public class BlockingSimulationCallable<T> implements Callable {

    /**
     * The Callable that should be run
     */
    private Callable<T> task;

    /**
     * The semaphor that should be notified if the thread has run successfully
     */
    private Semaphor notifier;

    /**
     * The thread that placed the order for this Callable
     */
    private Thread ancestor;

    /**
     * Create an instance of a BlockingSimulationCallable with a Callable to run
     * and a Semaphor that blocks the calling thread
     * @param taskToRun the task to run
     * @param lock the semaphor that gets notified after execution finished
     * @param ancestor
     */
    public BlockingSimulationCallable(Callable<T> taskToRun, Semaphor lock,
                                      Thread ancestor) {
        this.task = taskToRun;
        this.notifier = lock;
        this.ancestor = ancestor;
    }

    public Thread getAncestor() {
        return ancestor;
    }

    /**
     * This method should be called by the Executor only
     * @throws Exception
     */
    public FutureTask<T> call() throws Exception {
        FutureTask<T> returnValue = new FutureTask<T>(task);
        try {
            returnValue.run();
        } catch (Exception e) {
            throw e;
        } finally {
            notifier.V();
            ancestor = null;
        }
        return returnValue;
    }

    /**
     * Some error occured during the runtime of the FutureTask.
     * Reset the counter to free the semaphor
     */
    protected void abort() {
        notifier.V();
    }
}