/*
 * Created on Feb 12, 2004
 *
 * A NetComponentFigure is a virtual figure that hats no graphical
 * representation and no othe rfunction as to group a cople of other figures
 * into one loosely connected group. Mouse drags are being passed by such a
 * figure on to all its attached figures. Selections are being passed on to one
 * (or some) specific figures.
 */
package de.renew.netcomponents;

import CH.ifa.draw.DrawPlugin;

import CH.ifa.draw.application.DrawApplication;

import CH.ifa.draw.figures.AttributeFigure;
import CH.ifa.draw.figures.GroupHandle;
import CH.ifa.draw.figures.LineConnection;

import CH.ifa.draw.framework.ChildFigure;
import CH.ifa.draw.framework.DrawingView;
import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.framework.FigureChangeEvent;
import CH.ifa.draw.framework.FigureChangeListener;
import CH.ifa.draw.framework.FigureEnumeration;
import CH.ifa.draw.framework.Handle;

import CH.ifa.draw.standard.FigureEnumerator;
import CH.ifa.draw.standard.MergedFigureEnumerator;
import CH.ifa.draw.standard.RelativeLocator;

import CH.ifa.draw.util.ColorMap;
import CH.ifa.draw.util.StorableInput;
import CH.ifa.draw.util.StorableOutput;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.geom.Rectangle2D;

import java.io.IOException;
import java.io.ObjectInputStream;

import java.util.Collection;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Vector;


/**
 * The NetComponentFigure groups a set of net elements.
 * The set can be conveniently moved and selected (double-click) by mouse.
 *
 * @author Lawrence Cabac
 */
@SuppressWarnings("serial")
public class NetComponentFigure extends AttributeFigure
        implements FigureChangeListener {
    Vector<Figure> attached = new Vector<Figure>();

    public void drawFrame(Graphics g) {
        Rectangle r = displayBox();
        Graphics2D g2 = (Graphics2D) g;
        Shape s = new Rectangle2D.Float(r.x, r.y, r.width, r.height);
        g2.draw(s);
    }

    public void drawBackground(Graphics g) {
        Rectangle r = displayBox();
        Graphics2D g2 = (Graphics2D) g;
        Shape s = new Rectangle2D.Float(r.x, r.y, r.width, r.height);
        g2.fill(s);
    }


    /**
     * Group a vector of figures to form a NetComponentFigure.
     *
     * @param c -
     *            a collection of Figures that will be grouped.
     * @return the same collection
     */
    public Collection<Figure> group(Collection<Figure> c) {
        Iterator<Figure> it = c.iterator();
        while (it.hasNext()) {
            Figure figure = it.next();
            if ((!(figure instanceof ChildFigure))
                        || ((ChildFigure) figure).parent() == null) {
                if (!attached.contains(figure)) {
                    attached.add(figure);
                    figure.addFigureChangeListener(this);
                }
            }
        }

        return c;
    }

    protected void basicMoveBy(int dx, int dy) {
        Vector<Figure> selected = getView().selection();
        Iterator<Figure> it = attached.iterator();
        while (it.hasNext()) {
            AttributeFigure af = (AttributeFigure) it.next();
            if (!selected.contains(af)) {
                af.moveBy(dx, dy);
            }
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see CH.ifa.draw.standard.AbstractFigure#basicDisplayBox(java.awt.Point,
     *      java.awt.Point)
     */
    public void basicDisplayBox(Point origin, Point corner) {
        // TODO Auto-generated method stub
    }

    /**
     * Gets the display box. The display box is defined as the union of the
     * contained figures.
     */
    public Rectangle displayBox() {
        Enumeration<Figure> k = attached.elements();
        Rectangle r = new Rectangle();
        if (k.hasMoreElements()) {
            r = (k.nextElement()).displayBox();
        }
        while (k.hasMoreElements()) {
            r.add((k.nextElement()).displayBox());
        }
        if (r == null) {
            return new Rectangle(0, 0);
        }
        return r;
    }

    /*
     * (non-Javadoc)
     *
     * @see CH.ifa.draw.standard.AbstractFigure#handles()
     */


    /**
     * Gets the handles for the GroupFigure.
     */
    public Vector<Handle> handles() {
        Vector<Handle> handles = new Vector<Handle>();
        handles.addElement(new GroupHandle(this, RelativeLocator.northWest()));
        handles.addElement(new GroupHandle(this, RelativeLocator.northEast()));
        handles.addElement(new GroupHandle(this, RelativeLocator.southWest()));
        handles.addElement(new GroupHandle(this, RelativeLocator.southEast()));
        return handles;
    }

    /*
     * (non-Javadoc)
     *
     * @see CH.ifa.draw.framework.Figure#moveBy(int, int)
     */
    public void moveBy(int dx, int dy) {
        //super.moveBy(dx, dy);
        DrawingView view = getView();
        if (view == null) {
            return;
        }
        Vector<Figure> selected = view.selection();
        Iterator<Figure> it = attached.iterator();
        while (it.hasNext()) {
            Figure af = it.next();
            if (!selected.contains(af)) {
                af.moveBy(dx, dy);
            }
        }
        changed();
    }

    /**
     * Returns whether this figure can be selected.
     */
    public boolean isSelectable() {
        return true;
    }

    private DrawingView getView() {
        DrawApplication gui = DrawPlugin.getGui();
        if (gui == null) {
            return null;
        }
        return gui.getView(gui.drawing());

    }

    /**
     * Stores the connector and its owner to a StorableOutput.
     */
    public void write(StorableOutput dw) {
        super.write(dw);
        dw.writeInt(attached.size());
        Iterator<Figure> it = attached.iterator();
        while (it.hasNext()) {
            Figure figure = it.next();
            dw.writeStorable(figure);

        }
    }

    /**
     * Reads the connector and its owner from a StorableInput.
     */
    public void read(StorableInput dr) throws IOException {
        if (dr.getVersion() > 9) {
            super.read(dr);
        }
        attached = new Vector<Figure>();
        int size = dr.readInt();
        if (size != 0) {
            for (int i = 1; i <= size; i++) {
                Figure figure = (Figure) dr.readStorable();
                attached.add(figure);
                figure.addFigureChangeListener(this);
            }
        }


        // manage unset colors -- default for NetComponentFigure is ColoMap.NONE
        Enumeration<String> attributeKeys = getAttributeKeys();
        boolean isFillColorSet = false;
        boolean isFrameColorSet = false;
        while (attributeKeys.hasMoreElements()) {
            String string = (String) attributeKeys.nextElement();
            if (string.equals("FillColor")) {
                isFillColorSet = true;
            }
            if (string.equals("FrameColor")) {
                isFrameColorSet = true;
            }
        }
        if (!isFillColorSet) {
            setFillColor(ColorMap.NONE);
        }
        if (!isFrameColorSet) {
            setFrameColor(ColorMap.NONE);
            setAttribute("LineStyle", LineConnection.LINE_STYLE_DOTTED);
        }
    }

    /**
     * @return a <code>Vector</code> of Figures that are attached to this
     *         <code>NetComponentFigure</code>.
     */
    public Vector<Figure> getAttached() {
        return attached;
    }

    public boolean inspect(DrawingView view, boolean alternate) {
        if (alternate) {
            return super.inspect(view, alternate);
        } else {
            view.clearSelection();
            view.addToSelectionAll(attached);
            return true;
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see CH.ifa.draw.framework.FigureChangeListener#figureInvalidated(CH.ifa.draw.framework.FigureChangeEvent)
     */
    public void figureInvalidated(FigureChangeEvent e) {
        // nothing to do
    }

    /*
     * (non-Javadoc)
     *
     * @see CH.ifa.draw.framework.FigureChangeListener#figureChanged(CH.ifa.draw.framework.FigureChangeEvent)
     */
    public void figureChanged(FigureChangeEvent e) {
        invalidate();
        changed();
    }

    /*
     * (non-Javadoc)
     *
     * @see CH.ifa.draw.framework.FigureChangeListener#figureRemoved(CH.ifa.draw.framework.FigureChangeEvent)
     */
    public void figureRemoved(FigureChangeEvent e) {
        Figure removed = e.getFigure();
        attached.remove(removed);
        if (attached.isEmpty() && listener() != null) {
            listener().figureRequestRemove(new FigureChangeEvent(this));
        }
        invalidate();
        changed();
    }

    /*
     * (non-Javadoc)
     *
     * @see CH.ifa.draw.framework.FigureChangeListener#figureRequestRemove(CH.ifa.draw.framework.FigureChangeEvent)
     */
    public void figureRequestRemove(FigureChangeEvent e) {
        //attached.remove(e.getSource());
    }

    /*
     * (non-Javadoc)
     *
     * @see CH.ifa.draw.framework.FigureChangeListener#figureRequestUpdate(CH.ifa.draw.framework.FigureChangeEvent)
     */
    public void figureRequestUpdate(FigureChangeEvent e) {
        // nothing to do
    }

    /*
     * (non-Javadoc)
     *
     * @see CH.ifa.draw.framework.FigureChangeListener#figureHandlesChanged(CH.ifa.draw.framework.FigureChangeEvent)
     */
    public void figureHandlesChanged(FigureChangeEvent e) {
        // nothing to do
    }

    /**
     * Deserialization method, behaves like default readObject
     * method, but additionally restores the association from
     * contained figures to this composite figure.
     **/
    private void readObject(ObjectInputStream s)
            throws ClassNotFoundException, IOException {
        s.defaultReadObject();

        Enumeration<Figure> k = attached.elements();
        while (k.hasMoreElements()) {
            FigureEnumeration en = DrawPlugin.getGui().drawing().figures();
            Figure figure = k.nextElement();
            while (en.hasMoreElements()) {
                Figure f = en.nextElement();
                if (f.equals(figure)) {
                    figure.addToContainer(this);
                }
            }
        }
    }

    public FigureEnumeration getFiguresWithDependencies() {
        return new MergedFigureEnumerator(super.getFiguresWithDependencies(),
                                          getAttachedFigures());
    }


    /**
     * @return
     */
    private FigureEnumeration getAttachedFigures() {
        return new FigureEnumerator(attached);
    }


    /**
     *
     */
    public void updateListeners() {
        FigureEnumeration fenum = getFiguresWithDependencies();
        while (fenum.hasMoreElements()) {
            Figure figure = fenum.nextFigure();
            figure.addToContainer(this);
        }
    }
}