package de.renew.navigator;

import CH.ifa.draw.DrawPlugin;

import CH.ifa.draw.io.CombinationFileFilter;
import CH.ifa.draw.io.SimpleFileFilter;
import CH.ifa.draw.io.importFormats.ImportFormat;

import CH.ifa.draw.util.Iconkit;

import de.renew.plugin.IPlugin;
import de.renew.plugin.PluginManager;

import de.renew.shadow.DefaultShadowNetLoader;
import de.renew.shadow.DefaultShadowNetLoader.PathEntry;

import de.renew.util.StringUtil;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.event.MouseEvent;

import java.io.File;
import java.io.FileFilter;

import java.util.Enumeration;
import java.util.HashSet;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.ToolTipManager;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;


/**
 * The NavigatorGUI class contains all GUI related implementation for the
 * navigator. It is based upon a JFrame and setting up the interface, including
 * all viewable parts of the Navigator. The main component is the JTree,
 * containing all the opened FileTreeNodes.
 *
 * This class offers methods to access and manipulate the tree as well as
 * opening files in Renew. For this purpose, multiple node selection is
 * supported. These methods are mainly in use by the registered listeners of
 * this GUI.
 *
 * @author Hannes Ahrens (4ahrens)
 * @date March 2009
 */
public class NavigatorGUI extends JFrame {
    private static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                                                        .getLogger(NavigatorGUI.class);
    public static final long serialVersionUID = 97936353687387679L;
    public static String filesAtStartup = "";
    public static final String FILES_AT_STARTUP = "de.renew.navigator.filesAtStartup";
    public static String workspaceLocation = "";
    public static final String WORKSPACE_LOCATION = "de.renew.navigator.workspace";
    public static final String IMAGES = "/de/renew/navigator/images/";
    public static final String ICON_ADD = IMAGES
                                          + "eclipse-icons/obj16/add_obj.gif";
    public static final String ICON_COLLAPSEALL = IMAGES
                                                  + "eclipse-icons/elcl16/collapseall.gif";
    public static final String ICON_FILE_AIP = IMAGES
                                               + "application-x-renew-interactiondiagram.png";
    public static final String ICON_FILE_DEFAULT = IMAGES
                                                   + "plain-sheet-48x48.png";
    public static final String ICON_FILE_DRAW = IMAGES
                                                + "application-x-renew-drawing.png";
    public static final String ICON_FILE_RNW = IMAGES
                                               + "application-x-renew-net.png";
    public static final String ICON_HOME = IMAGES
                                           + "eclipse-icons/elcl16/home_nav.gif";
    public static final String ICON_NAVIGATOR = IMAGES
                                                + "eclipse-icons/eview16/filenav_nav.gif";
    public static final String ICON_NETPATH = IMAGES
                                              + "eclipse-icons/eview16/bkmrk_nav.gif";
    public static final String ICON_REFRESHALL = IMAGES
                                                 + "eclipse-icons/elcl16/refresh_nav.gif";
    public static final String ICON_REMOVEALL = IMAGES
                                                + "eclipse-icons/elcl16/progress_remall.gif";
    public static final String ICON_REMOVEONE = IMAGES
                                                + "eclipse-icons/elcl16/progress_rem.gif";
    private IPlugin _simulatorPlugin = null;
    private JPanel _mainPanel = null;
    private JPanel _menuPanel = null;
    private JScrollPane _scrollPane = null;
    private JPanel _treePanel = null;
    private JTree _tree = null;
    private DefaultMutableTreeNode _rootNode = null;
    private DefaultTreeModel _treeModel = null;
    private FileTreeCellRenderer _treeRenderer = null;
    private JPopupMenu _contextMenu = null;
    private TreeMouseListener _treeMouseListener = null;
    private TreeActionListener _treeActionListener = null;
    private TreeKeyListener _treeKeyListener = null;
    private File _previousFile = null;
    private CombinationFileFilter _cff = null;

    /**
     * The constructor only sets the name of the main frame of this navigator.
     */
    public NavigatorGUI() {
        super("(File-)Navigator");
    }

    /**
     * Initialization of the GUI. Here all the menus, buttons, the tree and all
     * viewable components of the navigator are created. Icons are loaded with
     * the CH.ifa.draw.util.Iconkit and action listeners are registred.
     */
    public void init() {
        this.setPreferredSize(new Dimension(250, 600));
        this.setLocation(0, 150);

        // get simulator plug-in
        _simulatorPlugin = PluginManager.getInstance()
                                        .getPluginsProviding("de.renew.simulator")
                                        .iterator().next();

        // get FileFilter
        _cff = DrawPlugin.getCurrent().getIOHelper().getFileFilter();
        ImportFormat[] importformats = DrawPlugin.getCurrent().getImportHolder()
                                                 .allImportFormats();
        for (ImportFormat importFormat : importformats) {
            if (logger.isDebugEnabled()) {
                logger.debug(NavigatorGUI.class.getSimpleName()
                             + ": import format "
                             + importFormat.fileFilter().getDescription());
            }
            if (importFormat.fileFilter() instanceof SimpleFileFilter) {
                _cff.add((SimpleFileFilter) importFormat.fileFilter());
                if (logger.isDebugEnabled()) {
                    logger.debug(NavigatorGUI.class.getSimpleName()
                                 + ": added "
                                 + importFormat.fileFilter().getDescription());
                }
            }
        }
        _cff.allowDirectory(true);
        _cff.allowHidden(false);
        if (logger.isDebugEnabled()) {
            HashSet<SimpleFileFilter> fileFilters = _cff.getFileFilters();
            for (SimpleFileFilter simpleFileFilter : fileFilters) {
                logger.debug(NavigatorGUI.class.getSimpleName() + ": "
                             + simpleFileFilter.getExtension());
            }
        }

        Iconkit.instance().registerImage(ICON_ADD);
        Iconkit.instance().registerImage(ICON_COLLAPSEALL);
        Iconkit.instance().registerImage(ICON_FILE_AIP);
        Iconkit.instance().registerImage(ICON_FILE_DEFAULT);
        Iconkit.instance().registerImage(ICON_FILE_DRAW);
        Iconkit.instance().registerImage(ICON_FILE_RNW);
        Iconkit.instance().registerImage(ICON_HOME);
        Iconkit.instance().registerImage(ICON_NAVIGATOR);
        Iconkit.instance().registerImage(ICON_NETPATH);
        Iconkit.instance().registerImage(ICON_REFRESHALL);
        Iconkit.instance().registerImage(ICON_REMOVEALL);
        Iconkit.instance().registerImage(ICON_REMOVEONE);
        this.setIconImage(Iconkit.instance().getImage(ICON_NAVIGATOR));

        _treeMouseListener = new TreeMouseListener(this);
        _treeActionListener = new TreeActionListener(this);
        _treeKeyListener = new TreeKeyListener(this);

        _mainPanel = new JPanel();
        _mainPanel.setLayout(new BoxLayout(_mainPanel, BoxLayout.PAGE_AXIS));
        this.add(_mainPanel);

        _menuPanel = new JPanel();
        _menuPanel.setLayout(new BoxLayout(_menuPanel, BoxLayout.LINE_AXIS));
        _menuPanel.add(Box.createHorizontalGlue());
        _mainPanel.add(_menuPanel);

        JButton menuButton = null;
        Dimension buttonDimension = new Dimension(33, 20);

        menuButton = new JButton(new ImageIcon(Iconkit.instance()
                                                      .getImage(ICON_HOME)));
        menuButton.setToolTipText("Open Home paths of Navigator properties");
        menuButton.setActionCommand("Open Home");
        menuButton.addActionListener(_treeActionListener);
        menuButton.setMinimumSize(buttonDimension);
        menuButton.setPreferredSize(buttonDimension);
        _menuPanel.add(menuButton);

        menuButton = new JButton(new ImageIcon(Iconkit.instance()
                                                      .getImage(ICON_NETPATH)));
        menuButton.setToolTipText("Open NetPaths of simulator properties");
        menuButton.setActionCommand("Open NetPaths");
        menuButton.addActionListener(_treeActionListener);
        menuButton.setMinimumSize(buttonDimension);
        menuButton.setPreferredSize(buttonDimension);
        _menuPanel.add(menuButton);

        menuButton = new JButton(new ImageIcon(Iconkit.instance()
                                                      .getImage(ICON_ADD)));
        menuButton.setToolTipText("Add Files (Ctrl+Shift+O)");
        menuButton.setActionCommand("Add Files");
        menuButton.addActionListener(_treeActionListener);
        menuButton.setMinimumSize(buttonDimension);
        menuButton.setPreferredSize(buttonDimension);
        _menuPanel.add(menuButton);

        menuButton = new JButton(new ImageIcon(Iconkit.instance()
                                                      .getImage(ICON_COLLAPSEALL)));
        menuButton.setToolTipText("Collapse All (Ctrl+Shift+C)");
        menuButton.setActionCommand("Collapse All");
        menuButton.addActionListener(_treeActionListener);
        menuButton.setMinimumSize(buttonDimension);
        menuButton.setPreferredSize(buttonDimension);
        _menuPanel.add(menuButton);

        menuButton = new JButton(new ImageIcon(Iconkit.instance()
                                                      .getImage(ICON_REFRESHALL)));
        menuButton.setToolTipText("Refresh All (Ctrl+R)");
        menuButton.setActionCommand("Refresh All");
        menuButton.addActionListener(_treeActionListener);
        menuButton.setMinimumSize(buttonDimension);
        menuButton.setPreferredSize(buttonDimension);
        _menuPanel.add(menuButton);

        menuButton = new JButton(new ImageIcon(Iconkit.instance()
                                                      .getImage(ICON_REMOVEONE)));
        menuButton.setToolTipText("Remove Node");
        menuButton.setActionCommand("Remove TreeNode");
        menuButton.addActionListener(_treeActionListener);
        menuButton.setMinimumSize(buttonDimension);
        menuButton.setPreferredSize(buttonDimension);
        _menuPanel.add(menuButton);

        menuButton = new JButton(new ImageIcon(Iconkit.instance()
                                                      .getImage(ICON_REMOVEALL)));
        menuButton.setToolTipText("Remove All (Ctrl+Shift+Delete)");
        menuButton.setActionCommand("Remove All");
        menuButton.addActionListener(_treeActionListener);
        menuButton.setMinimumSize(buttonDimension);
        menuButton.setPreferredSize(buttonDimension);
        _menuPanel.add(menuButton);

        _rootNode = new DefaultMutableTreeNode("rootNode");
        _treeModel = new DefaultTreeModel(_rootNode);

        _tree = new JTree(_treeModel) {
                private static final long serialVersionUID = 3105716467712346866L;

                public String getToolTipText(MouseEvent ev) {
                    if (ev == null) {
                        return null;
                    }
                    TreePath path = _tree.getPathForLocation(ev.getX(),
                                                             ev.getY());
                    if (path != null) {
                        return getTreeNodePath(path);
                    }
                    return null;
                }
            };
        ToolTipManager.sharedInstance().registerComponent(_tree);
        _tree.setRootVisible(false);
        _tree.setFocusable(true);
        // _tree.requestFocusInWindow();
        _tree.getSelectionModel()
             .setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
        // _tree.addTreeSelectionListener(this);
        _tree.addMouseListener(_treeMouseListener);
        _tree.addKeyListener(_treeKeyListener);
        _tree.setBorder(BorderFactory.createEmptyBorder(2, 5, 2, 0));

        _treeRenderer = new FileTreeCellRenderer();
        _treeRenderer.setDefaultIcon(new ImageIcon(Iconkit.instance()
                                                          .getImage(ICON_FILE_DEFAULT)
                                                          .getScaledInstance(18,
                                                                             -1,
                                                                             Image.SCALE_SMOOTH)));
        _treeRenderer.addIcon(".aip",
                              new ImageIcon(Iconkit.instance()
                                                   .getImage(ICON_FILE_AIP)
                                                   .getScaledInstance(18, -1,
                                                                      Image.SCALE_SMOOTH)));
        _treeRenderer.addIcon(".draw",
                              new ImageIcon(Iconkit.instance()
                                                   .getImage(ICON_FILE_DRAW)
                                                   .getScaledInstance(18, -1,
                                                                      Image.SCALE_SMOOTH)));
        _treeRenderer.addIcon(".rnw",
                              new ImageIcon(Iconkit.instance()
                                                   .getImage(ICON_FILE_RNW)
                                                   .getScaledInstance(18, -1,
                                                                      Image.SCALE_SMOOTH)));
        _treeRenderer.addIcon(".arm",
                              new ImageIcon(Iconkit.instance()
                                                   .getImage(ICON_FILE_RNW)
                                                   .getScaledInstance(18, -1,
                                                                      Image.SCALE_SMOOTH)));
        _treeRenderer.addIcon(".mad",
                              new ImageIcon(Iconkit.instance()
                                                   .getImage(ICON_FILE_RNW)
                                                   .getScaledInstance(18, -1,
                                                                      Image.SCALE_SMOOTH)));
        _treeRenderer.addIcon(".net",
                              new ImageIcon(Iconkit.instance()
                                                   .getImage(ICON_FILE_RNW)
                                                   .getScaledInstance(18, -1,
                                                                      Image.SCALE_SMOOTH)));
        _tree.setCellRenderer(_treeRenderer);

        _treePanel = new JPanel(new BorderLayout());
        _treePanel.setAlignmentX(LEFT_ALIGNMENT);
        _treePanel.setBackground(Color.WHITE);
        _treePanel.add(_tree, BorderLayout.LINE_START);
        _scrollPane = new JScrollPane(_treePanel);
        _scrollPane.getVerticalScrollBar().setUnitIncrement(10);
        _scrollPane.getHorizontalScrollBar().setUnitIncrement(10);
        _mainPanel.add(_scrollPane);
    }

    private String getTreeNodePath(TreePath path) {
        try {
            MutableFileTreeNode test = (MutableFileTreeNode) (path
                                           .getLastPathComponent());
            return test.getFile().getPath();
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * This method loads predefined directories from plugin properties,
     * filtered by the CombinationFileFilter from CH.ifa.draw.IOHelper.
     * Each file and directory gets added by the addFile()
     * method. The retrieved CombinationFileFilter will applied to each of these
     * files.
     *
     * @return files added?
     */
    public boolean addFilesFromProperties() {
        IPlugin plugin = PluginManager.getInstance().getPluginByName("Navigator");
        try {
            filesAtStartup = plugin.getProperties().getProperty(FILES_AT_STARTUP);
            workspaceLocation = plugin.getProperties()
                                      .getProperty(WORKSPACE_LOCATION);
            if (!workspaceLocation.endsWith("/")) {
                workspaceLocation += "/";
            }
            String[] filesInList = filesAtStartup.split(";");
            for (int i = 0; i < filesInList.length; i++) {
                // choose file or directory with JFileChooser
                String filesToAdd = filesInList[i];
                if (!filesToAdd.startsWith("/")) {
                    filesToAdd = workspaceLocation + filesToAdd;
                }
                File dir = new File(filesToAdd);
                if (dir.exists()) {
                    addFile(dir, _cff);
                }
                collapseAll();
            }
            return filesInList.length > 0;
        } catch (Exception e) {
//            addFiles();
        }
        return false;
    }

    /**
     * This method opens a JFileChooser window to choose files and directories
     * to add to the tree. The files get filtered by the CombinationFileFilter
     * retrieved from CH.ifa.draw.IOHelper. Each file and directory gets added
     * by the addFile() method. The retrieved CombinationFileFilter will applied
     * to each of these files. Furthermore, the latest path of the JFileChooser
     * gets stored in the IOHelper.
     */
    public void addFiles() {
        // choose file or directory with JFileChooser
        JFileChooser fc = new JFileChooser(DrawPlugin.getCurrent().getIOHelper()
                                                     .getCurrentDirectory(_previousFile != null
                                                                          ? _previousFile
                                                                          : null));
        fc.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
        fc.setFileFilter(_cff);
        fc.setFileHidingEnabled(true);
        fc.setMultiSelectionEnabled(true);
        int returnVal = fc.showOpenDialog(this);

        if (returnVal == JFileChooser.APPROVE_OPTION) {
            File[] files = fc.getSelectedFiles();

            if (files != null) {
                for (int i = 0; i < files.length; i++) {
                    addFile(files[i], _cff);
                }

                File parent = files[0].getParentFile();
                if (parent != null && parent.isDirectory()) {
                    DrawPlugin.getCurrent().getIOHelper().setLastPath(parent);
                }
                _previousFile = files[0];
            }
        }
    }

    /**
     * Adds a File by it's path name.
     *
     * @param path to file
     * @return true if added
     */
    public boolean addFile(String path) {
        File f = new File(path);
        return addFile(f, _cff);
    }

    /**
     * The addFile() method is used to add files as well as directories to the
     * internal root node. The additional filter will be used when loading each
     * of the nodes.
     *
     * @param file
     *            the root file/directory to add
     * @param ff
     *            the filter to be used
     * @return true if added
     */
    public boolean addFile(File file, FileFilter ff) {
        if (file.exists()) {
            Enumeration<?> children = _rootNode.children();
            boolean isNode = false;
            while (children.hasMoreElements()) {
                MutableFileTreeNode child = (MutableFileTreeNode) children
                                                .nextElement();
                if (child.getFile().getPath().equals(file.getPath())) {
                    isNode = true;
                    break;
                }
            }
            if (!isNode) {
                MutableFileTreeNode treeNode = new MutableFileTreeNode(file, ff);
                _treeModel.insertNodeInto(treeNode, _rootNode,
                                          _rootNode.getChildCount());
                TreePath treePath = new TreePath(treeNode.getPath());
                _tree.expandPath(treePath);
                _tree.scrollPathToVisible(treePath);
                _scrollPane.validate();
                _mainPanel.validate();
                _mainPanel.repaint();

                return true;
            }
        }
        return false;
    }

    /**
     * Removes a MutableTreeNode from the tree.
     *
     * @param mtn
     *            MutableTreeNode to remove
     */
    public void remove(MutableTreeNode mtn) {
        _treeModel.removeNodeFromParent(mtn);
    }

    /**
     * Removes a MutableTreeNode from the tree.
     *
     */
    public void remove() {
        TreePath treePath = _tree.getSelectionPath();
        if (treePath != null && treePath.getLastPathComponent() != null) {
            try {
                _treeModel.removeNodeFromParent((MutableTreeNode) treePath
                    .getLastPathComponent());
            } catch (Exception e) {
                System.out.println("Remove failed " + e.getMessage());
            }
        }
    }

    /**
     * This method tries to open a tree node of a specific position in Renew.
     * For opening the files, it is using the CH.ifa.draw.IOHelper.
     *
     * @param x
     *            tree relative x position of the node
     * @param y
     *            tree relative y position of the node
     */
    public void open(int x, int y) {
        TreePath path = _tree.getPathForLocation(x, y);
        if (path != null) {
            Object o = path.getLastPathComponent();
            if (o != null && o instanceof FileTreeNode) {
                FileTreeNode ftn = ((FileTreeNode) o);
                File file = ftn.getFile();
                if (!file.isDirectory()) {
                    DrawPlugin.getCurrent().getIOHelper()
                              .loadAndOpenDrawing(file);
                }
            }
        }
    }

    /**
     * Opens all selected files in Renew. For opening the files, it is using the
     * CH.ifa.draw.IOHelper.
     */
    public void openSelected() {
        TreePath[] paths = _tree.getSelectionPaths();
        for (int i = 0; i < _tree.getSelectionCount(); i++) {
            Object o = paths[i].getLastPathComponent();
            if (o != null && o instanceof FileTreeNode) {
                FileTreeNode ftn = ((FileTreeNode) o);
                File file = ftn.getFile();
                if (!file.isDirectory()) {
                    DrawPlugin.getCurrent().getIOHelper()
                              .loadAndOpenDrawing(file);
                }
            }
        }
    }

    /**
     * Closes all selected trees. Therefore it is checked if the selected nodes
     * are direct children of the internal root node.
     */
    public void closeSelected() {
        TreePath[] paths = _tree.getSelectionPaths();
        for (int i = 0; i < paths.length; i++) {
            // if(paths[i].getPathCount() == 2){
            Object o = paths[i].getLastPathComponent();
            if (o != null && o instanceof MutableTreeNode) {
                MutableTreeNode mtn = ((MutableTreeNode) o);
                if (mtn.getParent() == _rootNode) {
                    _treeModel.removeNodeFromParent(mtn);
                }
            }

            // }
        }
    }

    /**
     * This method pops up a simple context menu. It currently only contains a
     * selection field to remove the MutableTreeNode of the specified location.
     * The pop-up will only show if the node is a direct child to the internal
     * root node.
     *
     * @param x
     *            tree relative x position to show the pop-up
     * @param y
     *            tree relative y position to show the pop-up
     */
    public void showContextMenu(int x, int y) {
        TreePath path = _tree.getPathForLocation(x, y);
        if (path != null) {
            Object o = path.getLastPathComponent();
            if (o != null && o instanceof MutableTreeNode) {
                MutableTreeNode mtn = ((MutableTreeNode) o);
                if (mtn.getParent() == _rootNode) {
                    _contextMenu = new JPopupMenu();
                    JMenuItem menuItem = new JMenuItem("Remove Tree");
                    _treeActionListener.setTreeNode(mtn);
                    menuItem.addActionListener(_treeActionListener);
                    _contextMenu.add(menuItem);
                    _contextMenu.show(_tree, x, y);
                }
            }
        }
    }

    /**
     * closeAll() removes all opened nodes from tree.
     */
    public void closeAll() {
        _rootNode.removeAllChildren();
        _treeModel.nodeStructureChanged(_rootNode);
    }

    /**
     * collapseAll() collapses all visible nodes and invalidates the rest.
     */
    public void collapseAll() {
        _tree.collapsePath(new TreePath(_rootNode));
        _treeModel.nodeStructureChanged(_rootNode);
    }

    /**
     * refreshALL() refreshes the whole JTree. The method checks all loaded
     * (displayed) nodes for file existence. Expanded nodes get refreshed and
     * children of collapsed nodes get invalidated.
     */
    public void refreshAll() {
        for (int i = _tree.getRowCount() - 1; i >= 0; i--) {
            TreePath path = _tree.getPathForRow(i);
            Object o = path.getLastPathComponent();
            if (o != null && o instanceof MutableFileTreeNode) {
                MutableFileTreeNode node = (MutableFileTreeNode) o;
                refresh(node);
            }
        }
    }

    /**
     * This method compares the node's children file names with the files
     * on file system and updates all children to represent current
     * state of file system.
     *
     * @return have changes occurred?
     */
    public boolean refresh(MutableFileTreeNode node) {
        if (!node.exists()) {
            _treeModel.removeNodeFromParent(node);
            return true;
        } else if (node.getAllowsChildren()) {
            if (_tree.isExpanded(new TreePath(node.getPath()))) {
                if (node.getChildCount() == 0) {
                    if (node.reloadChildren()) {
                        _treeModel.nodeStructureChanged(node);
                        return true;
                    }
                } else {
                    File[] files = node.getChildFiles();
                    node.ensureCapacity(files.length);
                    TreeNode tn = null;
                    int i = 0;
                    boolean bChanged = false;

                    // remove all non existent files
                    while (i < node.getChildCount()) {
                        tn = node.getChildAt(i);
                        if (tn instanceof MutableFileTreeNode) {
                            if (!((MutableFileTreeNode) tn).getFile().exists()) {
                                _treeModel.removeNodeFromParent((MutableFileTreeNode) tn);
                                bChanged = true;
                                continue;
                            }
                        }
                        i++;
                    }

                    // add new files
                    for (i = 0; i < files.length; i++) {
                        boolean bFound = false;
                        int pos = -1;
                        for (int j = 0; j < node.getChildCount(); j++) {
                            tn = node.getChildAt(j);
                            if (tn instanceof MutableFileTreeNode) {
                                int comp = files[i].compareTo(((MutableFileTreeNode) tn)
                                                              .getFile());
                                if (comp == 0) {
                                    bFound = true;
                                    break;
                                } else if (comp < 0 && pos == -1) {
                                    pos = j;
//                                    break;
                                }
                            }
                        }

                        if (!bFound) {
                            if (pos < 0) {
                                pos = node.getChildCount();
                            }
                            MutableFileTreeNode mftn = new MutableFileTreeNode(node,
                                                                               files[i],
                                                                               _cff);
                            _treeModel.insertNodeInto(mftn, node, pos);
                            bChanged = true;
                        }
                    }
                    return bChanged;
                }
            } else {
                if (node.invalidateChildren()) {
                    _treeModel.nodeStructureChanged(node);
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Open the Home path, set by Navigator properties.
     */
    public void openHome() {
        closeAll();
        if (!addFilesFromProperties()) {
            addFile(new File(System.getProperty("user.dir")), _cff);
        }
    }

    /**
     * Open the NetPaths, set by simulator plugin properties.
     */
    public void openNetPaths() {
        closeAll();
        PathEntry[] entries = DefaultShadowNetLoader.canonizePaths(StringUtil
                                  .splitPaths(_simulatorPlugin.getProperties()
                                                              .getProperty("de.renew.netPath")));
        for (int i = 0; i < entries.length; i++) {
            if (!entries[i].isClasspathRelative) {
                addFile(entries[i].path);
            }
        }
    }
}