package de.renew.navigator;

import java.io.File;
import java.io.FileFilter;

import java.util.Collections;
import java.util.Comparator;

import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;


/**
 * The MutableFileTreeNode class is extending the FileTreeNode class and implementing the MutableTreeNode interface.
 * It is virtually overwriting the loadChildren method, to get them loading as MutableFileTreeNodes instead of FileTreeNodes.
 * This class allows dynamically inserting MutableTreeNodes, removing them and setting the class's parent node.
 * It supports refreshing it's children to reflect external changes in file system, without reloading the whole tree.
 *
 * @author Hannes Ahrens (4ahrens)
 * @date March 2009
 */
public class MutableFileTreeNode extends FileTreeNode implements MutableTreeNode {

    /**
     * @see de.renew.navigator.FileTreeNode#FileTreeNode(File)
     */
    public MutableFileTreeNode(File file) {
        super(file);
        sortChildren();
    }

    /**
     * @see de.renew.navigator.FileTreeNode#FileTreeNode(File, FileFilter)
     */
    public MutableFileTreeNode(File file, FileFilter ff) {
        super(file, ff);
        sortChildren();
    }

    /**
     * @see de.renew.navigator.FileTreeNode#FileTreeNode(TreeNode, File)
     */
    public MutableFileTreeNode(FileTreeNode parent, File file) {
        super(parent, file);
        sortChildren();
    }

    /**
     * @see de.renew.navigator.FileTreeNode#FileTreeNode(TreeNode, File, FileFilter)
     */
    public MutableFileTreeNode(FileTreeNode parent, File file, FileFilter ff) {
        super(parent, file, ff);
        sortChildren();
    }

    /**
     * @see javax.swing.tree.MutableTreeNode#insert(javax.swing.tree.MutableTreeNode, int)
     */
    public void insert(MutableTreeNode child, int index) {
        _children.add(index, child);
        sortChildren();
    }

    /**
     * @see javax.swing.tree.MutableTreeNode#remove(int)
     */
    public void remove(int index) {
        _children.remove(index);
    }

    /**
     * @see javax.swing.tree.MutableTreeNode#remove(javax.swing.tree.MutableTreeNode)
     */
    public void remove(MutableTreeNode node) {
        _children.remove(node);
    }

    /**
     * @see javax.swing.tree.MutableTreeNode#removeFromParent()
     */
    public void removeFromParent() {
        if (_parent != null && _parent instanceof MutableTreeNode) {
            ((MutableTreeNode) _parent).remove(this);
        }
    }

    /**
     * @see javax.swing.tree.MutableTreeNode#setParent(javax.swing.tree.MutableTreeNode)
     */
    public void setParent(MutableTreeNode newParent) {
        _parent = newParent;
    }

    /**
     * This method currently is doing nothing.
     * @see javax.swing.tree.MutableTreeNode#setUserObject(java.lang.Object)
     */
    public void setUserObject(Object object) {
    }

    /**
     * This method overwrites the corresponding virtual method of FileTreeNode.
     * This assures that all the children get loaded as MutableFileTreeNodes!
     *
     * @param files the files to be loaded as new children
     */
    protected void loadChildren(File[] files) {
        for (int i = 0; i < files.length; i++) {
            _children.add(new MutableFileTreeNode(this, files[i], _ff));
        }
    }

    private void sortChildren() {
        reloadChildren();
        if (this._children != null) {
            Collections.sort(this._children,
                             new Comparator<TreeNode>() {
                    public int compare(TreeNode o1, TreeNode o2) {
                        return o1.toString().compareToIgnoreCase(o2.toString());
                    }
                });
        }
    }
}