package de.renew.navigator;

import java.io.File;
import java.io.FileFilter;

import java.util.Enumeration;
import java.util.LinkedList;
import java.util.Vector;

import javax.swing.tree.TreeNode;


/**
 * The FileTreeNode class is a basic TreeNode implementation for Files.
 * It loads its children on request, can reload the children but doesn't offer methods to add or remove Children directly.
 * For using the DeaultTreeModel, the MutableTreeNode interface has to be implemented.
 * This is done by the MutableFileTreeNode class which is extending this class.
 * Cause this class is loading it's children as FileTreeNodes, the protected loadChildren method has to be overwritten on extension of this class.
 *
 * @author Hannes Ahrens (4ahrens)
 * @date March 2009
 */
public class FileTreeNode implements TreeNode {
    protected TreeNode _parent = null;
    protected Vector<TreeNode> _children = null;
    protected File _node;
    protected FileFilter _ff = null;
    protected boolean _bChildrenLoaded = false;

    /**
     * This constructor creates a root node.
     *
     * @param file the file to be contained
     */
    public FileTreeNode(File file) {
        assert (file != null);
        _node = file;
    }

    /**
     * This constructor creates a root node.
     * All children will get filtered and initialized with same filter.
     *
     * @param file the file to be contained
     * @param ff the filter to be used on loading the children.
     */
    public FileTreeNode(File file, FileFilter ff) {
        assert (file != null);
        _node = file;
        _ff = ff;
    }

    /**
     * This constructor creates a child node.
     *
     * @param parent the parent TreeNode
     * @param file the file to be contained
     */
    public FileTreeNode(TreeNode parent, File file) {
        assert (file != null);
        _parent = parent;
        _node = file;
    }

    /**
     * This constructor creates a child node.
     * All children will get filtered and initialized with same filter.
     *
     * @param parent the parent TreeNode
     * @param file the file to be contained
     * @param ff the filter to be used on loading the children.
     */
    public FileTreeNode(TreeNode parent, File file, FileFilter ff) {
        assert (file != null);
        _parent = parent;
        _node = file;
        _ff = ff;
    }

    /**
     * @see javax.swing.tree.TreeNode#children()
     */
    public Enumeration<TreeNode> children() {
        if (!_bChildrenLoaded) {
            reloadChildren();
        }
        return _children.elements();
    }

    /**
     * Only directories allow children in a file system.
     * @see javax.swing.tree.TreeNode#getAllowsChildren()
     *
     * @return is contained file a directory?
     */
    public boolean getAllowsChildren() {
        return _node.isDirectory();
    }

    /**
     * @see javax.swing.tree.TreeNode#getChildAt(int)
     */
    public TreeNode getChildAt(int childIndex) {
        if (!_bChildrenLoaded) {
            reloadChildren();
        }
        return _children.elementAt(childIndex);
    }

    /**
     * @see javax.swing.tree.TreeNode#getChildCount()
     */
    public int getChildCount() {
        if (!_bChildrenLoaded) {
            reloadChildren();
        }
        return _children.size();
    }

    /**
     *
     */
    public void ensureCapacity(int capacity) {
        _children.ensureCapacity(capacity);
    }

    /**
     * @see javax.swing.tree.TreeNode#getIndex(javax.swing.tree.TreeNode)
     */
    public int getIndex(TreeNode node) {
        return _children.indexOf(node);
    }

    /**
     * @see javax.swing.tree.TreeNode#getParent()
     */
    public TreeNode getParent() {
        return _parent;
    }

    /**
     * This method is giving the tree path from the root node.
     * @see javax.swing.tree.DefaultMutableTreeNode#getPath()
     *
     * @return all parent TreeNodes + this
     */
    public TreeNode[] getPath() {
        LinkedList<TreeNode> path = new LinkedList<TreeNode>();
        TreeNode node = this;
        while (node != null) {
            path.addFirst(node);
            node = node.getParent();
        }
        return path.toArray(new TreeNode[] {  });
    }

    /**
     * Checks whether a file is already in child list.
     *
     * @param f file to look for
     * @return true if file is contained
     */
    public boolean isChild(File f) {
        for (int i = 0; i < _children.size(); i++) {
            TreeNode node = _children.get(i);
            if (node instanceof FileTreeNode) {
                if (((FileTreeNode) node).getFile().getPath().equals(f.getPath())) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * All non directory files are leafs.
     * @see javax.swing.tree.TreeNode#isLeaf()
     *
     * @return is no directory
     */
    public boolean isLeaf() {
        return !_node.isDirectory();
    }

    /**
     * @return is file existent
     */
    public boolean exists() {
        return _node.exists();
    }

    /**
     * This method reloads the children of this class from file system.
     * For initializing the child node, the loadChildren method gets invoked with the new files.
     *
     * @return true if files loaded
     */
    public boolean reloadChildren() {
        _bChildrenLoaded = true;
        if (_node.isDirectory()) {
            File[] files = _ff == null ? _node.listFiles() : _node.listFiles(_ff);
            if (_children == null) {
                _children = new Vector<TreeNode>(files.length);
            } else {
                _children.clear();
                _children.ensureCapacity(files.length);
            }
            if (files.length > 0) {
                loadChildren(files);
                return true;
            }
        }
        return false;
    }

    /**
     * This method is only for internal use and for virtual overload.
     * Classes extending the FileTreeNode should overwrite this method, so the children won't be loaded as FileTreeNodes.
     *
     * @param files the files to be loaded as new children
     */
    protected void loadChildren(File[] files) {
        for (int i = 0; i < files.length; i++) {
            _children.add(new FileTreeNode(this, files[i], _ff));
        }
    }

    /**
     * @return the contained file of this class
     */
    public File getFile() {
        return _node;
    }

    /**
     * @return current FileFilter
     */
    public FileFilter getFilter() {
        return _ff;
    }

    /**
     * @return the filtered child files of this node
     */
    public File[] getChildFiles() {
        return _ff == null ? _node.listFiles() : _node.listFiles(_ff);
    }

    /**
     * The toString method get's used by the JTree to determine the text for representing a node.
     * Therefore the filename gets returned.
     *
     * @return the name of this node
     */
    public String toString() {
        return _node.getName();
    }

    /**
     * Invalidating the children of this node, clears them and causes them to reload on next access to a child.
     *
     * @return have children been dropped?
     */
    public boolean invalidateChildren() {
        boolean bResult = _children != null && _children.size() > 0;
        if (_children != null) {
            _children.clear();
        }
        _children = null;
        _bChildrenLoaded = false;
        return bResult;
    }

    /**
     * @return returns the file extension ".*" of this file if available, else ""
     */
    public String getFileExtension() {
        String name = _node.getName();
        int index = name.lastIndexOf('.');
        if (index == -1) {
            return "";
        } else {
            return name.substring(index);
        }
    }
}