package de.renew.navigator;

import java.awt.Component;

import java.util.HashMap;

import javax.swing.Icon;
import javax.swing.JTree;
import javax.swing.tree.DefaultTreeCellRenderer;


/**
 * The FileTreeCellRenderer class can be applied to a JTree as CellRenderer.
 * It extends the DefaultTreeCellRenderer with the ability to store and render different Icons for different file types.
 *
 * @author Hannes Ahrens (4ahrens)
 * @date March 2009
 */
public class FileTreeCellRenderer extends DefaultTreeCellRenderer {
    private static final long serialVersionUID = -7788334592931402079L;
    private HashMap<String, Icon> _iconMap = null;
    private Icon _defaultIcon = null;

    /**
     * The default constructor i initializing the internal icon HashMap.
     */
    public FileTreeCellRenderer() {
        _iconMap = new HashMap<String, Icon>();
    }

    /**
     * @return the default Icon or null if none set
     */
    public Icon getDefaultIcon() {
        return _defaultIcon;
    }

    /**
     * @param fileIcon the Icon to set as default
     */
    public void setDefaultIcon(Icon fileIcon) {
        _defaultIcon = fileIcon;
    }

    /**
     * @param key the file extension ".*" the icon shall be displayed for
     * @param fileIcon the Icon to render
     * @return previous fileIcon if key already being used, else null
     */
    public Icon addIcon(String key, Icon fileIcon) {
        return _iconMap.put(key, fileIcon);
    }

    /**
     * @param key the file extension ".*" to remove the Icon from
     * @return the Icon being removed or null if none available at the specified key position
     */
    public Icon removeIcon(String key) {
        return _iconMap.remove(key);
    }

    /**
     * This method overwrites the corresponding virtual method of the DefaultTreeCellRenderer.
     * It's used by the JTree to render its nodes and sets the corresponding Icon for each node when rendered.
     * @see javax.swing.tree.DefaultTreeCellRenderer#getTreeCellRendererComponent(JTree, Object, boolean, boolean, boolean, int, boolean)
     *
     * @return the component used for drawing the specified Node (Object)
     */
    public Component getTreeCellRendererComponent(JTree tree, Object value,
                                                  boolean sel,
                                                  boolean expanded,
                                                  boolean leaf, int row,
                                                  boolean hasFocus) {
        super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf,
                                           row, hasFocus);
        if (leaf && value instanceof FileTreeNode) {
            Icon fileIcon = _iconMap.get(((FileTreeNode) value).getFileExtension()
                                          .toLowerCase());
            if (fileIcon != null) {
                setIcon(fileIcon);
            } else if (_defaultIcon != null) {
                setIcon(_defaultIcon);
            }
        }
        return this;
    }
}