/*
 * Created on Nov 22, 2004
 *
 */
package de.renew.gui.logging;

import de.renew.application.SimulatorPlugin;

import de.renew.engine.common.SimulatorEvent;
import de.renew.engine.common.StepIdentifier;

import java.util.Vector;


/**
 * @author Sven Offermann
 *
 */
public class MainRepositoryManager {
    private static MainRepositoryManager manager = null;

    // manages the repositories for different simulation runs.
    // for performance reasons this is implemented with arrays 
    // and not with util classes.
    private long[] simulationRunIds;
    private MainRepository[] repositories;

    /** simulationHistoriesDim determines the dimension of the
     * simulation history. This value determines how many
     * simulation traces should be stored in the memory.
     * The simulation traces are stored in a kind of ring puffer.
     * Older simulations traces will be removed from the puffer.
     * This value should be as small as possible to prevent
     * memory consumption.
     */
    private static int defaultSimulationHistoriesDim = 2;

    public MainRepositoryManager() {
        this(defaultSimulationHistoriesDim);
    }

    public MainRepositoryManager(int simulationHistoriesDim) {
        this.simulationRunIds = new long[simulationHistoriesDim];
        this.repositories = new MainRepository[simulationRunIds.length];
    }

    public static MainRepositoryManager getInstance() {
        if (manager == null) {
            manager = new MainRepositoryManager();
        }

        return manager;
    }

    public long[] getSimulationRunIds() {
        Vector<Long> idVector = new Vector<Long>();

        int x = 0;
        while ((x < this.simulationRunIds.length)
                       && (this.repositories[x] != null)) {
            idVector.add(new Long(this.simulationRunIds[x]));
            x++;
        }

        long[] ids = new long[idVector.size()];
        for (x = 0; x < idVector.size(); x++) {
            ids[x] = idVector.get(x).longValue();
        }

        return ids;
    }

    public MainRepository getRepository(long simulationRunId) {
        int x = 0;
        while ((x < repositories.length) && (repositories[x] != null)) {
            if (simulationRunIds[x] == simulationRunId) {
                // found the queried repository 
                return repositories[x];
            }
            x++;
        }

        // cant find an existing repository, so i will create a new one.
        MainRepository repository = new MainRepository();


        // add repository with the runId to the list of known repositories.
        // For performance reasons we shift the existing repositories by one
        // position in the array list, cause normally the last created repository
        // will be queried most.
        for (int y = (repositories.length - 1); y > 0; y--) {
            simulationRunIds[y] = simulationRunIds[y - 1];
            repositories[y] = repositories[y - 1];
        }

        simulationRunIds[0] = simulationRunId;
        repositories[0] = repository;

        return repository;
    }

    public MainRepository getRepository(SimulatorEvent simEvent) {
        StepIdentifier step = simEvent.getStep();

        //Simulation may be terminated while getting the repository
        if (step != null) {
            long runId = step.getSimulationRunId();

            return getRepository(runId);
        }
        return null;

    }

    public StepTraceRepository getCurrentRepository(String loggerName) {
        long currentSimId = SimulatorPlugin.getCurrent().getCurrentEnvironment()
                                           .getSimulator()
                                           .currentStepIdentifier()
                                           .getSimulationRunId();

        return getRepository(currentSimId).getLoggerRepository(loggerName, -1);
    }
}