/*
 * Created on Aug 23, 2004
 */
package de.renew.gui.logging;

import de.renew.engine.common.SimulatorEvent;
import de.renew.engine.common.StepIdentifier;

import java.util.Hashtable;
import java.util.Map;
import java.util.Vector;


/**
 * Manages the log messages for the different simulation loggers for one simulation run.
 *
 * @author Sven Offermann
 */
public class MainRepository extends AbstractRepository
        implements StepTraceChangeListener {

    /**
    * maps logger names to LoggerRepositories
    */
    private Map<String, LoggerRepository> loggerRepositories = new Hashtable<String, LoggerRepository>();

    /**
     * maps StepIdentifiers to StepTraces (stores all occured log messages
     * for a single simulation step)
     */
    private Map<StepIdentifier, StepTrace> stepTraces = new Hashtable<StepIdentifier, StepTrace>();

    public LoggerRepository getLoggerRepository(String loggerName, int capacity) {
        LoggerRepository lRepository = this.loggerRepositories.get(loggerName);

        if (lRepository == null) {
            lRepository = new LoggerRepository(this, capacity);
            this.loggerRepositories.put(loggerName, lRepository);
            this.addRepositoryChangeListener(lRepository);
        }

        lRepository.setCapacity(capacity);
        return lRepository;
    }

    public void addEvent(SimulatorEvent event) {
        // add the event information to the repository
        StepTrace stepTrace = this.stepTraces.get(event.getStep());
        if (stepTrace == null) {
            stepTrace = new StepTrace(event.getStep());
            stepTrace.addStepTraceChangeListener(this);
            this.stepTraces.put(event.getStep(), stepTrace);
            this.fireStepTraceAdded(stepTrace);
        }

        stepTrace.log(event.getMessage());
    }

    public StepTrace[] getStepTraces(StepIdentifier[] steps) {
        Vector<StepTrace> traces = new Vector<StepTrace>();

        for (int x = 0; x < steps.length; x++) {
            StepTrace trace = stepTraces.get(steps[x]);
            if (trace != null) {
                traces.add(trace);
            }
        }

        return traces.toArray(new StepTrace[] {  });
    }

    public StepTrace[] getAllStepTraces() {
        return stepTraces.entrySet().toArray(new StepTrace[] {  });
    }

    public StepTrace getStepTrace(StepIdentifier stepIdentifier) {
        StepTrace trace = this.stepTraces.get(stepIdentifier);

        return trace;
    }

    public boolean removeStepTrace(StepIdentifier stepIdentifier) {
        StepTrace stepTrace = getStepTrace(stepIdentifier);

        if (stepTrace != null) {
            if (!(fireStepTraceRemoveRequest(stepTrace))) {
                // no vetos, so we can remove the StepTrace from the repository
                this.stepTraces.remove(stepIdentifier);

                return true;
            }
        }

        return false;
    }

    public void stepTraceChanged(StepTrace stepTrace) {
        // simply forward the change event
        this.fireStepTraceChanged(stepTrace);
    }
}