/*
 * Created on 19.08.2004
 */
package de.renew.gui.logging;

import CH.ifa.draw.framework.DrawingView;
import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.framework.FigureWithID;

import de.renew.application.SimulatorPlugin;

import de.renew.engine.events.NetEvent;
import de.renew.engine.events.PlaceEvent;
import de.renew.engine.events.SimulationEvent;
import de.renew.engine.events.TransitionEvent;

import de.renew.gui.CPNApplication;
import de.renew.gui.CPNInstanceDrawing;
import de.renew.gui.GuiPlugin;

import de.renew.net.NetElementID;

import de.renew.remote.NetInstanceAccessor;
import de.renew.remote.PlaceInstanceAccessor;
import de.renew.remote.RemotePlugin;
import de.renew.remote.TransitionInstanceAccessor;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.ListSelectionModel;
import javax.swing.border.EtchedBorder;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;


/**
 * The table model for the different defined loggers, which holds the tables
 * with the trace messages from the simulator steps.
 *
 * @author Sven Offermann
 */
public class LoggerTableModel extends TableModel
        implements RepositoryChangeListener {
    private static final org.apache.log4j.Logger logger = org.apache.log4j.Logger
                                                          .getLogger(LoggerTableModel.class);
    private boolean permanentUpdate = true;
    private Set<StepTraceChangeListener> listeners = new HashSet<StepTraceChangeListener>();

    /**
     * the StepTraceRepository represented by this table model.
     */
    private StepTraceRepository repository;

    /**
     * stores the stepTraces in the same order like in the Repository. If a
     * stepTrace is removed from the appender, this information is used to
     * determine the row index of the removed StepTrace.
     */
    private List<StepTrace> traces = new ArrayList<StepTrace>();

    /**
     * Creates a new table model to display the logged messages of the given
     * GuiAppender.
     *
     * @param loggerName
     *            the GuiAppender represented by this table model
     */
    public LoggerTableModel(String loggerName) {
        super(true);

        this.repository = MainRepositoryManager.getInstance()
                                               .getCurrentRepository(loggerName);

        if (repository != null) {
            repository.addRepositoryChangeListener(this);

            StepTrace[] stepTraces = repository.getAllStepTraces();
            for (int x = 0; x < stepTraces.length; x++) {
                addStepTrace(stepTraces[x]);
            }
        }
    }

    public void setPermanentUpdate(boolean update) {
        this.permanentUpdate = update;
    }

    private void addStepTrace(StepTrace stepTrace) {
        if (stepTrace.getStepIdentifier().getComponents().length > 0) {
            JPanel panel = new JPanel(new GridLayout(1, 1));
            StepTableModel stepModel = new StepTableModel(stepTrace);
            JTable tb = new JTable(stepModel);

            ErrorLevelTableRenderer elr = new ErrorLevelTableRenderer();
            tb.setDefaultRenderer(Object.class, elr);

            stepModel.addTableModelListener(new TableModelListenerImpl(stepTrace));

            // create popupMenu
            tb.addMouseListener(new PopupMenuMouseListener());
            panel.setBorder(BorderFactory.createTitledBorder(BorderFactory
                .createEtchedBorder(EtchedBorder.RAISED), stepTrace.toString()));
            panel.add(tb);

            addRow(new Object[] { panel });
            this.traces.add(stepTrace);

            fireTableRowsInserted(getRowCount() - 1, getRowCount() - 1);
        }
    }

    private void removeStepTrace(StepTrace stepTrace) {
        if (stepTrace.getStepIdentifier().getComponents().length > 0) {
            int row = this.traces.indexOf(stepTrace);
            if (row != -1) {
                removeRow(row);
                this.traces.remove(row);

                if (this.permanentUpdate) {
                    this.fireTableRowsDeleted(row, row);
                }
            }
        }
    }

    public void stepTraceChanged(StepTrace stepTrace) {
        if (permanentUpdate == true) {
            fireStepTraceChanged(stepTrace);
        }
    }

    public void stepTraceAdded(StepTraceRepository repository,
                               StepTrace stepTrace) {
        if (permanentUpdate == true) {
            addStepTrace(stepTrace);
            fireStepTraceAdded(repository, stepTrace);
        }
    }

    public void stepTraceRemoved(StepTraceRepository repository,
                                 StepTrace stepTrace) {
        if (permanentUpdate == true) {
            removeStepTrace(stepTrace);
            fireStepTraceRemoved(repository, stepTrace);
        }
    }

    public void stepTraceRemoveRequest(StepTraceRemoveRequest request) {
        // do nothing
        // the decision if a StepTrace can be removed, is done in the
        // LoggerRepository or MainRepository
    }

    public int getIndexOf(StepTrace stepTrace) {
        return this.traces.indexOf(stepTrace);
    }

    public void addRepositoryChangeListener(StepTraceChangeListener listener) {
        this.listeners.add(listener);
    }

    public void fireStepTraceChanged(StepTrace stepTrace) {
        RepositoryChangeListener[] l = this.listeners.toArray(new RepositoryChangeListener[] {  });
        for (int x = 0; x < l.length; x++) {
            l[x].stepTraceChanged(stepTrace);
        }
    }

    public void fireStepTraceAdded(StepTraceRepository repository,
                                   StepTrace stepTrace) {
        RepositoryChangeListener[] l = this.listeners.toArray(new RepositoryChangeListener[] {  });
        for (int x = 0; x < l.length; x++) {
            l[x].stepTraceAdded(repository, stepTrace);
        }
    }

    public void fireStepTraceRemoved(StepTraceRepository repository,
                                     StepTrace stepTrace) {
        RepositoryChangeListener[] l = this.listeners.toArray(new RepositoryChangeListener[] {  });
        for (int x = 0; x < l.length; x++) {
            l[x].stepTraceRemoved(repository, stepTrace);
        }
    }

    public void fireStepTraceRemoveRequest(StepTraceRemoveRequest request) {
        RepositoryChangeListener[] l = this.listeners.toArray(new RepositoryChangeListener[] {  });
        for (int x = 0; x < l.length; x++) {
            l[x].stepTraceRemoveRequest(request);
        }
    }

    // --------------------------------------------------------------------
    private class PopupMenuMouseListener extends MouseAdapter {
        public void mousePressed(MouseEvent e) {
            Component c = e.getComponent();
            if ((c instanceof JTable) && (e.isPopupTrigger())) {
                openPopup((JTable) c, e);
            }
        }

        public void mouseReleased(MouseEvent e) {
            Component c = e.getComponent();
            if ((c instanceof JTable) && (e.isPopupTrigger())) {
                openPopup((JTable) c, e);
            }
        }

        /*
         * (non-Javadoc)
         *
         * @see
         * java.awt.event.MouseAdapter#mouseClicked(java.awt.event.MouseEvent)
         */
        public void mouseClicked(MouseEvent e) {
            if (e.getClickCount() == 2) {
                Component c = e.getComponent();
                if (c instanceof JTable) {
                    // open a new window with a detailed view of the simulation
                    // event log message
                    openDetailView((JTable) c, e);
                }
            }
        }

        private void openPopup(JTable table, MouseEvent e) {
            Point p = e.getPoint();
            int row = table.rowAtPoint(p);
            ListSelectionModel selectionModel = table.getSelectionModel();
            selectionModel.setSelectionInterval(row, row);
            StepTableModel model = (StepTableModel) table.getModel();

            SimulationEvent simEvent = (SimulationEvent) model.getValueAt(row, 0);

            JPopupMenu popup = new JPopupMenu();
            if (SimulatorPlugin.getCurrent().isSimulationActive()) {
                RemotePlugin remote = RemotePlugin.getInstance();
                JMenuItem item1 = new JMenuItem("show net pattern");
                if (!(simEvent instanceof NetEvent)) {
                    item1.setEnabled(false);
                } else {
                    item1.addActionListener(new OpenNetPatternAction(remote
                        .wrapInstance(((NetEvent) simEvent).getNetInstance())));
                }
                popup.add(item1);

                JMenuItem item2 = new JMenuItem("show net instance");
                if (!(simEvent instanceof NetEvent)) {
                    item2.setEnabled(false);
                } else {
                    item2.addActionListener(new OpenNetInstanceAction(remote
                        .wrapInstance(((NetEvent) simEvent).getNetInstance())));
                }
                popup.add(item2);

                popup.addSeparator();

                JMenuItem item3 = new JMenuItem("show net pattern element");
                if ((simEvent instanceof PlaceEvent)
                            || (simEvent instanceof TransitionEvent)) {
                    if (simEvent instanceof PlaceEvent) {
                        item3.addActionListener(new OpenNetPatternElementAction(remote
                                                                                .wrapInstance(((PlaceEvent) simEvent)
                                                                                              .getPlaceInstance())));
                    } else {
                        item3.addActionListener(new OpenNetPatternElementAction(remote
                                                                                .wrapInstance(((TransitionEvent) simEvent)
                                                                                              .getTransitionInstance())));
                    }
                } else {
                    item3.setEnabled(false);
                }
                popup.add(item3);

                JMenuItem item4 = new JMenuItem("show net instance element");
                if ((simEvent instanceof PlaceEvent)
                            || (simEvent instanceof TransitionEvent)) {
                    if (simEvent instanceof PlaceEvent) {
                        item4.addActionListener(new OpenNetInstanceElementAction(remote
                                                                                 .wrapInstance(((PlaceEvent) simEvent)
                                                                                               .getPlaceInstance())));
                    } else {
                        item4.addActionListener(new OpenNetInstanceElementAction(remote
                                                                                 .wrapInstance(((TransitionEvent) simEvent)
                                                                                               .getTransitionInstance())));
                    }
                } else {
                    item4.setEnabled(false);
                }
                popup.add(item4);
            } else {
                JMenuItem item5 = new JMenuItem("Simulation terminated");
                item5.setEnabled(false);
                popup.add(item5);
            }

            popup.show(table, e.getX(), e.getY());
        }

        private void openDetailView(JTable table, MouseEvent e) {
            Point p = e.getPoint();
            int row = table.rowAtPoint(p);
            StepTableModel model = (StepTableModel) table.getModel();
            SimulationEvent simEvent = (SimulationEvent) model.getValueAt(row, 0);

            JFrame detailFrame = new JFrame();
            detailFrame.setSize(600, 300);
            detailFrame.getContentPane().setLayout(new BorderLayout());
            JTextArea textArea = new JTextArea(simEvent.toString());
            JScrollPane scrollPane = new JScrollPane(textArea);
            detailFrame.getContentPane().add(scrollPane, BorderLayout.CENTER);

            detailFrame.setVisible(true);
        }

        // Inner action classes.
        private class OpenNetInstanceAction implements ActionListener {
            private NetInstanceAccessor netInstance;

            public OpenNetInstanceAction(NetInstanceAccessor instance) {
                netInstance = instance;
            }

            public void actionPerformed(ActionEvent e) {
                GuiPlugin.getCurrent().getGui().openInstanceDrawing(netInstance);
            }
        }

        private class OpenNetPatternAction implements ActionListener {
            private NetInstanceAccessor netInstance;

            public OpenNetPatternAction(NetInstanceAccessor instance) {
                netInstance = instance;
            }

            public void actionPerformed(ActionEvent e) {
                try {
                    GuiPlugin.getCurrent().getGui()
                             .openNetPatternDrawing(netInstance.getNet()
                                                               .getName());
                } catch (Exception ex) {
                    logger.error(ex.getMessage(), ex);
                }
            }
        }

        private class OpenNetPatternElementAction implements ActionListener {
            private Object elementInstance;

            public OpenNetPatternElementAction(PlaceInstanceAccessor instance) {
                elementInstance = instance;
            }

            public OpenNetPatternElementAction(TransitionInstanceAccessor instance) {
                elementInstance = instance;
            }

            public void actionPerformed(ActionEvent e) {
                try {
                    NetInstanceAccessor netInstance = null;
                    NetElementID elementID = null;
                    if (elementInstance instanceof PlaceInstanceAccessor) {
                        PlaceInstanceAccessor placeInstance = (PlaceInstanceAccessor) elementInstance;
                        netInstance = placeInstance.getNetInstance();
                        elementID = placeInstance.getID();
                    } else if (elementInstance instanceof TransitionInstanceAccessor) {
                        TransitionInstanceAccessor transitionInstance = (TransitionInstanceAccessor) elementInstance;
                        netInstance = transitionInstance.getNetInstance();
                        elementID = transitionInstance.getID();
                    }
                    int figureID = FigureWithID.NOID;
                    if (elementID != null) {
                        figureID = elementID.getFigureID();
                    }
                    GuiPlugin guiPlugin = GuiPlugin.getCurrent();
                    CPNApplication gui = guiPlugin.getGui();

                    // NOTICEnull
                    if (netInstance != null) {
                        String netName = netInstance.getNet().getName();
                        gui.openNetPatternDrawing(netName, figureID);
                    }
                } catch (Exception ex) {
                    logger.error(ex.getMessage(), ex);
                }
            }
        }

        private class OpenNetInstanceElementAction implements ActionListener {
            private Object elementInstance;

            public OpenNetInstanceElementAction(PlaceInstanceAccessor instance) {
                elementInstance = instance;
            }

            public OpenNetInstanceElementAction(TransitionInstanceAccessor instance) {
                elementInstance = instance;
            }

            public void actionPerformed(ActionEvent e) {
                try {
                    NetInstanceAccessor netInstance = null;
                    NetElementID elementID = null;
                    if (elementInstance instanceof PlaceInstanceAccessor) {
                        PlaceInstanceAccessor placeInstance = (PlaceInstanceAccessor) elementInstance;
                        netInstance = placeInstance.getNetInstance();
                        elementID = placeInstance.getID();
                    } else if (elementInstance instanceof TransitionInstanceAccessor) {
                        TransitionInstanceAccessor transitionInstance = (TransitionInstanceAccessor) elementInstance;
                        netInstance = transitionInstance.getNetInstance();
                        elementID = transitionInstance.getID();
                    }
                    int figureID = FigureWithID.NOID;
                    if (elementID != null) {
                        figureID = elementID.getFigureID();
                    }
                    GuiPlugin guiPlugin = GuiPlugin.getCurrent();
                    CPNApplication gui = guiPlugin.getGui();
                    gui.openInstanceDrawing(netInstance);
                    CPNInstanceDrawing instDwg = CPNInstanceDrawing
                                                     .getInstanceDrawing(netInstance);
                    if ((instDwg != null) && (figureID != FigureWithID.NOID)) {
                        Figure figure = instDwg.getInstanceFigureOfFigureWithID(figureID);
                        DrawingView view = gui.getView(instDwg);
                        if (figure != null && view != null) {
                            view.clearSelection();
                            view.addToSelection(figure);
                            view.repairDamage();
                        }
                    }
                } catch (Exception ex) {
                    logger.error(ex.getMessage(), ex);
                }
            }
        }
    }

    // implementation of the TableModelListener
    private class TableModelListenerImpl implements TableModelListener {
        private StepTrace stepTrace;

        public TableModelListenerImpl(StepTrace stepTrace) {
            this.stepTrace = stepTrace;
        }

        public void tableChanged(TableModelEvent e) {
            if ((e.getType() == TableModelEvent.INSERT)
                        || (e.getType() == TableModelEvent.UPDATE)) {
                fireTableCellUpdated(getIndexOf(this.stepTrace), 0);
            }
        }
    }
}