/*
 * Created on 26.11.2004
 *
 */
package de.renew.gui.logging;

import de.renew.engine.common.SimulatorEvent;
import de.renew.engine.common.StepIdentifier;

import java.util.List;
import java.util.Vector;


/**
 * @author Sven Offermann
 *
 */
public class LoggerRepository extends AbstractRepository
        implements RepositoryChangeListener {
    private int capacity = 0;
    private List<StepIdentifier> stepTraces = new Vector<StepIdentifier>();
    private MainRepository repository;

    public LoggerRepository(MainRepository repository, int capacity) {
        if (capacity >= 0) {
            this.capacity = capacity;
        }
        this.repository = repository;
    }

    public void setCapacity(int capacity) {
        if (capacity >= 0) {
            this.capacity = capacity;
        }
    }

    public StepTrace[] getStepTraces(StepIdentifier[] steps) {
        Vector<StepIdentifier> traces = new Vector<StepIdentifier>();
        for (int x = 0; x < steps.length; x++) {
            if (stepTraces.contains(steps[x])) {
                traces.add(steps[x]);
            }
        }

        return repository.getStepTraces(traces.toArray(new StepIdentifier[] {  }));
    }

    public StepTrace[] getAllStepTraces() {
        return repository.getStepTraces(stepTraces.toArray(new StepIdentifier[] {  }));
    }

    public StepTrace getStepTrace(StepIdentifier stepIdentifier) {
        if (!stepTraces.contains(stepIdentifier)) {
            return null;
        }

        return repository.getStepTrace(stepIdentifier);
    }

    public boolean removeStepTrace(StepIdentifier stepIdentifier) {
        this.stepTraces.remove(stepIdentifier);

        StepTrace stepTrace = repository.getStepTrace(stepIdentifier);

        // remove from the underlying repository
        repository.removeStepTrace(stepIdentifier);

        // so we must inform our listeners about the removal of the StepTrace.
        this.fireStepTraceRemoved(stepTrace);

        return true;
    }

    public void addEvent(SimulatorEvent event) {
        // add messages to the underliing repository.
        repository.addEvent(event);

        if (!this.stepTraces.contains(event.getStep())) {
            if ((capacity > 0) && (stepTraces.size() > capacity)) {
                while (stepTraces.size() > capacity) {
                    StepIdentifier rStep = stepTraces.get(0);
                    removeStepTrace(rStep);
                }
            }

            this.stepTraces.add(event.getStep());

            // inform listeners that we added some logging information
            this.fireStepTraceAdded(repository.getStepTrace(event.getStep()));
        }
    }

    public void stepTraceAdded(StepTraceRepository repository,
                               StepTrace stepTrace) {
        // forward only if we know the added StepTrace. Normally we added it.
        if (this.stepTraces.contains(stepTrace.getStepIdentifier())) {
            this.fireStepTraceAdded(stepTrace);
        }
    }

    public void stepTraceRemoved(StepTraceRepository repository,
                                 StepTrace stepTrace) {
        // do nothing. we inform our listeners in the remove method.
    }

    public void stepTraceRemoveRequest(StepTraceRemoveRequest request) {
        if (this.stepTraces.contains(request.getStepTrace().getStepIdentifier())) {
            request.veto();
        } else {
            // forward the event to registered listeners
            this.fireStepTraceRemoveRequest(request.getStepTrace());
        }
    }

    public void stepTraceChanged(StepTrace stepTrace) {
        // forward only if we know the added StepTrace. Normally we added it.
        if (this.stepTraces.contains(stepTrace.getStepIdentifier())) {
            this.fireStepTraceChanged(stepTrace);
        }
    }
}