/*
 * (c) Copyright 2001 MyCorporation.
 * All Rights Reserved.
 */
package de.renew.plugin.load;

import de.renew.plugin.IPlugin;
import de.renew.plugin.PluginAdapter;
import de.renew.plugin.PluginClassLoader;
import de.renew.plugin.PluginManager;
import de.renew.plugin.PluginProperties;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;

import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;


/**
 * Singleton support class that loads static plugins.
 *
 * There are two ways to instantiate a plugin with this class' loadPlugin(PluginProperties) method,
 * depending on what a call to getURL() on the given properties object returns:
 * if it is a jar file, this jar file will be assumed to contain the plugin code
 * as well as the plugin.cfg file with the plugin's configuration information;
 * if it is a directory, the directory is searched for the plugin.cfg file as well as
 * for a jarfile which is then used as the plugin code source.
 * If there is no jarfile in the directory, the directory itself is used as code source for
 * the resulting plugin.
 */
public class SimplePluginLoader implements PluginLoader {
    public static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                                                       .getLogger(SimplePluginLoader.class);

    /*
     * Initializes the PluginLoader instance.
     */
    public SimplePluginLoader() {
    }

    /**
     * Loads the plugin in the given directory.
     * For that, the plugin.cfg file is parsed and searched for
     * the pluginClassName entry.
     * This class will then be loaded and instantiated.
     */
    public IPlugin loadPlugin(PluginProperties props) {
        URL pluginLocation = props.getURL();
        logger.debug("SimplePluginLoader loading from " + props.getURL());
        URL[] pluginJars = unifyURL(pluginLocation);
        PluginClassLoader pluginClassLoader = PluginManager.getInstance()
                                                           .getPluginClassLoader();
        String message = "ok.";

        Set<URL> urls = new HashSet<URL>();
        for (URL url : pluginClassLoader.getURLs()) {
            urls.add(url);
        }

        IPlugin plugin = null;
        try {
            for (int x = 0; x < pluginJars.length; x++) {
                if (!urls.contains(pluginJars[x])) {
                    pluginClassLoader.addURL(pluginJars[x]);
                }
            }

            String className = props.getProperty("mainClass");

            if (className != null) {
                logger.debug("* creating a " + className + " with cl "
                             + pluginClassLoader);
                Class<?> clazz = pluginClassLoader.loadClass(className);
                if (clazz.getClassLoader() != pluginClassLoader) {
                    logger.warn("system classloader was used to load " + clazz
                                + "; it was probably in your classpath.");
                    logger.warn("That might turn out as a problem.");
                }

                Class<?>[] params = { props.getClass() };
                Constructor<?> cons = clazz.getConstructor(params);
                Object[] args = { props };
                plugin = (IPlugin) cons.newInstance(args);

                logger.debug("SimplePluginLoader: instantiated " + plugin);
            } else {
                logger.debug("* no main class!");
                plugin = new PluginAdapter(props);
            }
        } catch (InvocationTargetException ite) {
            Throwable e = ite.getTargetException();
            message = e.toString();
        } catch (Exception e) {
            message = e.toString();
        }
        if (plugin == null) {
            String desc = props.getName();
            if (desc == null) {
                desc = "from " + pluginLocation;
            }
            logger.error("SimplePluginLoader: Could not load plugin " + desc
                         + ": " + message);
        } else {
            logger.debug("SimplePluginLoader result: " + message);
        }
        return plugin;
    }

    public IPlugin loadPluginFromURL(URL url) {
        PluginProperties props = new PluginProperties(url);
        try {
            InputStream stream = PluginConfigFinder.getConfigInputStream(url);
            props.load(stream);
            return loadPlugin(props);
        } catch (MalformedURLException e) {
            logger.error("SimplePluginLoader.loadPluginFromURL: " + e);
        } catch (IOException e) {
            logger.error("SimplePluginLoader.loadPluginFromURL: " + e);
        }
        return null;
    }

    public static URL[] getURLsfromDirectory(File location) {
        // changed (06.05.2004) to support more than one jar in a plugin folder
        if (location == null) {
            return null;
        } else if (location.getName().endsWith(".jar")) {
            try {
                // look into jar file if there are additional lib jars 
                // includes
                Vector<URL> urls = new Vector<URL>();
                urls.add(location.toURI().toURL());

                String baseURL = "jar:" + location.toURI().toURL() + "!/";
                JarFile jar = new JarFile(location);
                Enumeration<JarEntry> e = jar.entries();
                while (e.hasMoreElements()) {
                    JarEntry entry = e.nextElement();
                    if ((entry.getName().startsWith("libs/"))
                                && (entry.getName().endsWith(".jar"))) {
                        urls.add(new URL(baseURL + entry.getName()));
                    }
                }

                return urls.toArray(new URL[] {  });
            } catch (MalformedURLException e) {
                logger.error("SimplePluginLoader: Could not convert to URL: "
                             + location + " (" + e.getMessage() + ").");
            } catch (IOException e1) {
                logger.error("Error while opening/reading jar file: "
                             + location, e1);
            }
        }

        if (!location.isDirectory()) {
            location = location.getParentFile();
        }

        return getJarsRecursiveFromDir(location).toArray(new URL[] {  });
    }

    public static URL[] unifyURL(URL given) {
        File dir = null;
        try {
            dir = new File(given.toURI());
        } catch (URISyntaxException e) {
            logger.warn("SimplePluginLoader: Unable to search for jar files in "
                        + given + ": " + e);
        } catch (IllegalArgumentException e) {
            logger.warn("SimplePluginLoader: Unable to search for jar files in "
                        + given + ": " + e);
        } catch (NullPointerException e) {
            logger.warn("SimplePluginLoader: Unable to search for jar files in "
                        + given + ": " + e);
        }
        URL[] jarURLs = getURLsfromDirectory(dir);
        if (jarURLs == null || jarURLs.length == 0) {
            logger.warn("SimplePluginLoader: No jar found in " + given
                        + ", resorting to given URL.");
            return new URL[] { given };
        }
        return jarURLs;
    }

    /**
     * Converts a String into a list by separating by commas
     */
    protected Collection<String> parseListString(String list) {
        StringTokenizer tok = new StringTokenizer(list, ",");
        Collection<String> result = new Vector<String>(tok.countTokens());
        try {
            for (int i = 0; tok.hasMoreTokens(); i++) {
                String currentToken = tok.nextToken();
                result.add(currentToken);
            }
        } catch (ArrayIndexOutOfBoundsException e) {
            logger.error("PluginLoader: " + e + " when parsing " + list
                         + " as list!");
        }
        return result;
    }

    private static Vector<URL> getJarsRecursiveFromDir(File dir) {
        Vector<URL> v = new Vector<URL>();

        File[] files = dir.listFiles();
        for (int x = 0; x < files.length; x++) {
            if (files[x].isFile()) {
                if (files[x].getName().endsWith(".jar")) {
                    try {
                        v.add(files[x].toURL());
                    } catch (MalformedURLException e) {
                        logger.debug("Can't convert file location to URL: "
                                     + e.getMessage(), e);
                    }
                }
            } else if (files[x].isDirectory()) {
                v.addAll(getJarsRecursiveFromDir(files[x]));
            }
        }

        return v;
    }

    public static class PluginConfigFinder {
        public static InputStream getConfigInputStream(URL pluginLocation)
                throws MalformedURLException, IOException {
            URL configLocation;
            configLocation = new URL(pluginLocation, "plugin.cfg");
            //		logger.debug ("loading cfg from " + configLocation.toExternalForm());
            InputStream stream = null;
            try {
                stream = configLocation.openStream();
            } catch (Exception e) {
                // maybe its a jar. try on.
                String urlString = pluginLocation.toExternalForm();
                int index = urlString.lastIndexOf('!');
                if (index > -1) {
                    urlString = urlString.substring(0, index) + "/plugin.cfg";
                    logger.debug("trying alternate URL for plugin.cfg: "
                                 + urlString);
                }
                configLocation = new URL(urlString);
                stream = configLocation.openStream();
            }
            return stream;
        }
    }

    public static class PluginSourceFinder {
        public static URL getPluginSource(URL pluginLocation) {
            return pluginLocation;
        }
    }
}