package de.renew.plugin.command;

import de.renew.plugin.CollectionLister;
import de.renew.plugin.IPlugin;
import de.renew.plugin.PluginManager;

import java.io.PrintStream;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;


/**
 * This command unloads the plugin with the given name.
 * If the argument "-v" is given, verbose output is printed,
 * especially concerning load errors regarding dependency problems.
 *
 * @author J&ouml;rn Schumacher
 */
public class UnloadCommand implements CLCommand {
    public static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                                                       .getLogger(UnloadCommand.class);

    public void execute(String[] args, PrintStream response) {
        if (args.length == 1
                    && ("h".equals(args[0]) || "-h".equals(args[0])
                               || "--help".equals(args[0])
                               || "--h".equals(args[0])
                               || "-help".equals(args[0]))) {
            response.println("usage: unload [-v | -r | -m] <PluginName>\n"
                             + "Description: \n"
                             + "use parameter -v to see a list of all dependent plug-ins\n"
                             + "use parameter -r to unload all dependent plug-ins recursivly\n"
                             + "use parameter -m to remove given plug-ins iff there are no dependors and\n"
                             + "\t\t at least one further plug-in which provides the same service (EXPERIMENTAL!!!)");
            return;
        }
        boolean verbose = false;
        boolean recursive = false;
        boolean allowMultipleServices = false;
        ArrayList<Integer> ignoreIndices = new ArrayList<Integer>();
        for (int i = 0; i < args.length; i++) {
            if (args[i].equals("-v")) {
                verbose = true;
                ignoreIndices.add(new Integer(i));
            } else if (args[i].equals("-r")) {
                recursive = true;
                ignoreIndices.add(new Integer(i));
            } else if (args[i].equals("-m")) {
                allowMultipleServices = true;
                ignoreIndices.add(new Integer(i));
            }
        }
        String pluginName = mergeArgs(args, ignoreIndices);

        if (pluginName == null || pluginName.trim().length() == 0) {
            response.println("UnloadCommand: please set name of plug-in!");
            return;
        }

        IPlugin toShutDown = PluginManager.getInstance()
                                          .getPluginByName(pluginName);
        if (toShutDown == null) {
            response.println("UnloadCommand: did not find plug-in with name "
                             + pluginName);
            return;
        }

        try {
            Map<String, Collection<IPlugin>> deps = checkDependencies(toShutDown,
                                                                      allowMultipleServices);
            Set<Entry<String, Collection<IPlugin>>> entries = deps.entrySet();
            if (entries.isEmpty()) {
                PluginManager.getInstance().stop(toShutDown);
                response.println("Plug-in " + toShutDown
                                 + " successfully removed.");
            } else if (recursive) {
                List<IPlugin> dependers = getDependers(toShutDown,
                                                       allowMultipleServices);
                response.println("recursively unloading "
                                 + CollectionLister.toString(dependers));
                PluginManager.getInstance().stop(dependers);
                response.println("Dependent plug-ins \n(" + dependers
                                 + ")\n successfully removed.");
                response.println("Plug-in " + toShutDown
                                 + " successfully removed.");
            } else {
                response.println("UnloadCommand: could not unload "
                                 + toShutDown
                                 + ": there are plug-ins depending on it.");
                if (verbose) {
                    Iterator<Entry<String, Collection<IPlugin>>> entryIt = entries
                                                                           .iterator();
                    while (entryIt.hasNext()) {
                        Entry<String, Collection<IPlugin>> ent = entryIt.next();
                        response.print(ent.getKey() + ":\t");
                        response.println(CollectionLister.toString(ent.getValue()));
                    }
                } else {
                    response.println("Use \"unload -v <PluginName>\" to see a list.");
                }
            }
        } catch (ArrayIndexOutOfBoundsException e) {
        } catch (NumberFormatException e) {
            response.println("cleanup canceled.");
        } catch (Exception e) {
            response.println("cleanup canceled: " + e + "; " + e.getMessage());
            logger.error(e.getMessage(), e);
        }
    }

    private String mergeArgs(String[] args, ArrayList<Integer> ignoreIndices) {
        String result = "";
        for (int i = 0; i < args.length; i++) {
            if (!ignoreIndices.contains(new Integer(i))) {
                if (!"".equals(result)) {
                    result += " ";
                }
                result += args[i];
            }
        }
        return result;
    }

    /**
     * Check whether the given plugin may be unloaded with regards to
     * require/depend-relation.
     * Returns a list of plugins that are dependent on the given plugin.
     */
    private Map<String, Collection<IPlugin>> checkDependencies(IPlugin toCheck,
                                                               boolean allowMultipleServices) {
        Collection<String> provisions = toCheck.getProperties().getProvisions();
        HashMap<String, Collection<IPlugin>> dependers = new HashMap<String, Collection<IPlugin>>();
        if (provisions != null) {
            Iterator<String> prIt = provisions.iterator();
            while (prIt != null && prIt.hasNext()) {
                String prov = prIt.next();
                boolean getAnother = false;
                Collection<IPlugin> all = PluginManager.getInstance()
                                                       .getPluginsProviding(prov);
                if (all != null && allowMultipleServices) {
                    Iterator<IPlugin> plugins = all.iterator();
                    while (plugins != null && plugins.hasNext()) {
                        IPlugin iPlugin = plugins.next();
                        if (!iPlugin.getName().equals(toCheck.getName())) {
                            getAnother = true;
                            break;
                        }
                    }
                }
                if (!getAnother) {
                    Collection<IPlugin> reqs = getRequirers(prov);
                    if (!reqs.isEmpty()) {
                        dependers.put(prov, reqs);
                    }
                }
            }
        }
        return dependers;
    }

    /*
     * These two methods call themselves, recursively,
     * to find all plugins that are
     */
    private List<IPlugin> getDependers(IPlugin toCheck,
                                       boolean allowMultipleServices) {
        List<IPlugin> result = new ArrayList<IPlugin>();
        result.add(toCheck);
        Map<String, Collection<IPlugin>> dependers = checkDependencies(toCheck,
                                                                       allowMultipleServices);
        Collection<Collection<IPlugin>> deps = dependers.values();
        result.addAll(getDependers(deps, allowMultipleServices));
        return result;
    }

    private List<IPlugin> getDependers(Collection<?> plugins,
                                       boolean allowMultipleServices) {
        List<IPlugin> result = new ArrayList<IPlugin>();
        Iterator<?> it = plugins.iterator();
        while (it.hasNext()) {
            Object o = it.next();
            List<IPlugin> toAdd;
            if (o instanceof Collection<?>) {
                Collection<?> toCheck = (Collection<?>) o;
                toCheck.removeAll(result);
                toAdd = getDependers(toCheck, allowMultipleServices);
            } else {
                IPlugin toCheck = (IPlugin) o;
                toAdd = getDependers(toCheck, allowMultipleServices);
            }

            // make sure the depending plugins are not added more than once
            result.removeAll(toAdd);
            result.addAll(toAdd);
        }
        return result;
    }

    private Collection<IPlugin> getRequirers(String prov) {
        Collection<IPlugin> result = new ArrayList<IPlugin>();
        Collection<IPlugin> plugins = PluginManager.getInstance().getPlugins();

        Iterator<IPlugin> plIt = plugins.iterator();

        // collect all plugins that require the current provision
        while (plIt.hasNext()) {
            IPlugin pl = plIt.next();
            Collection<String> reqs = pl.getProperties().getRequirements();
            if (reqs.contains(prov)) {
                result.add(pl);
            }
        }
        return result;
    }

    public String getDescription() {
        return "cleans up (deactivates) a plugin";
    }
}