/**
 *
 */
package de.renew.imagenetdiff;

import org.apache.log4j.Logger;

import CH.ifa.draw.framework.Drawing;

import CH.ifa.draw.io.StatusDisplayer;

import de.renew.io.exportFormats.PNGExportFormat;

import de.renew.plugin.PluginManager;

import de.renew.util.StringUtil;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;

import java.io.File;
import java.io.IOException;

import javax.imageio.ImageIO;

import javax.swing.JFrame;
import javax.swing.JScrollPane;


/**
 * Command to Export images to png and run compare (ImageMagick) on them.
 * Needs installed ImageMagick in the system.
 * Produces and overwrites (without warning) png files from selected drawings.
 * Produces png image as difference image of both. Results receives the same
 * name as the first selected image with ".diff.png" extension.
 * Images are not erased after execution.
 *
 * @author Lawrence Cabac
 *
 */
public class PNGDiffCommand extends AbstractDiffCommand implements DiffExecutor {
    private static final String IMAGE_EXTENSION = ".png";
    private static final Logger logger = Logger.getLogger(PNGDiffCommand.class);

    public PNGDiffCommand() {
        super("Simple PNG Diff");
    }

    /**
     * @param sd
     * @param drawing1
     * @param drawing2
     */
    @Override
    public void doDiff(StatusDisplayer sd, Drawing drawing1, Drawing drawing2,
                       boolean quite) {
        if (drawing1 != null && drawing2 != null) {
            // try exporting drawings to png
            // and run compare on them
            PNGExportFormat export = new PNGExportFormat();
            String name1 = StringUtil.stripFilenameExtension(drawing1.getFilename()
                                                                     .getAbsolutePath());
            String name2 = StringUtil.stripFilenameExtension(drawing2.getFilename()
                                                                     .getAbsolutePath());
            String nameDiffNoExt = name1 + "-diff";
            String nameDiff = name1 + "-diff" + IMAGE_EXTENSION;
            Rectangle bounds1 = drawing1.getBounds();
            Rectangle bounds2 = drawing2.getBounds();
            int width = Math.max(bounds1.width + bounds1.x,
                                 bounds2.width + bounds2.x);
            int height = Math.max(bounds1.height + bounds1.y,
                                  bounds2.height + bounds2.y);
            Rectangle bounds = new Rectangle(0, 0, width, height);

            // do the export
            Process process = null;
            try {
                export.internalExport(drawing1,
                                      new File(name1 + IMAGE_EXTENSION),
                                      bounds, false); // no clipping of white boarder (up/left)
                export.internalExport(drawing2,
                                      new File(name2 + IMAGE_EXTENSION),
                                      bounds, false); //no clipping
            } catch (Exception e1) {
                logger.error(e1.getMessage());
                logger.debug(e1.getMessage(), e1);
            }

            //NOTICEredundant
            int exit = 0;
            if (process != null) {
                try {
                    exit = process.waitFor();
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }

            exchangeColor(name1, 10, "white", "white");
            exchangeColor(name2, 10, "white", "white");

            if (drawing1.getFilename().getName().endsWith(".aip")
                        || drawing1.getFilename().getName().endsWith(".rnw")) {
                logger.debug("Doing AIP/RNW conversion.");
                exchangeColor(name1, 10, "lightgray", "white");
                exchangeColor(name1, 10, "lightgreen", "white");
                exchangeColor(name2, 10, "lightgray", "white");
                exchangeColor(name2, 10, "lightgreen", "white");
                exchangeColor(name1, 10, "seagreen1", "white");
                exchangeColor(name2, 10, "seagreen1", "white");
                exchangeColor(name1, 10, "yellow", "white");
                exchangeColor(name2, 10, "yellow", "white");
            }

            // compare the two images with im
            try {
                process = Runtime.getRuntime()
                                 .exec("compare " + name1 + IMAGE_EXTENSION
                                       + " " + name2 + IMAGE_EXTENSION + " "
                                       + nameDiff); // newer version of im knows options: -highlight-color and -lowlight-color 
            } catch (Exception e1) {
                logger.error(e1.getMessage());
                logger.debug(e1.getMessage(), e1);
            }
            exit = 0; //NOTICEredundant
            if (process != null) {
                try {
                    exit = process.waitFor();
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }

//            exchangeColor(nameDiff, 60, "#f54056","blue");
            process = null;
            try {
                exchangeColor(nameDiffNoExt, 10, "#d81030", "green");
//                process = Runtime.getRuntime()
//                                 .exec("convert " + nameDiff + " "
//                                       + "  -fuzz 60%  -fill blue  -opaque '#f54056' "
//                                       + nameDiff);
            } catch (Exception e1) {
                logger.error(e1.getMessage());
                logger.debug(e1.getMessage(), e1);
            }
            exit = 0; //NOTICEredundant
            if (process != null) {
                try {
                    exit = process.waitFor();
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }

            // Finally show diff image if exists, if no quite flag is set.
            File diffFile = new File(nameDiff);
            if (!diffFile.exists()) {
                sd.showStatus("Could not create diff image. ImageMagick installed?");
            } else if (!quite) {
                final ImageNetDiffPlugin plugin = (ImageNetDiffPlugin) PluginManager.getInstance()
                                                                                    .getPluginByName("ImageNetDiff");
                plugin.addBlock();
                final JFrame f = new JFrame("Diff Image: " + drawing1.getName()
                                            + " and " + drawing2.getName());

                f.addWindowListener(new WindowAdapter() {
                        public void windowClosing(WindowEvent e) {
                            f.dispose();
                            plugin.removeBlock();
                        }
                    });

                BufferedImage img = null;
                try {
                    img = ImageIO.read(diffFile);
                } catch (IOException e1) {
                    e1.printStackTrace();
                }
                ImageComponent comp = new ImageComponent(img, width, height);
                JScrollPane pane = new JScrollPane(comp);
                f.add(pane);
                f.setSize(new Dimension(width, height));
                f.pack();
                f.setVisible(true);
            } else {
                sd.showStatus("Diff image created successfully. Name: "
                              + diffFile.getName());
            }
        } else {
            sd.showStatus("Operation canceled.");
        }
    }

    private static void exchangeColor(String name, int fuzz, String oldColor,
                                      String newColor) {
        logger.debug("exchangecolor called with: " + fuzz + " " + oldColor
                     + " " + newColor);
        Process process;
        int exit;
        process = null;
        try {
            String cvrtstr = "convert " + name + IMAGE_EXTENSION + " "
                             + "-fuzz " + fuzz + "% -fill " + newColor
                             + "  -opaque " + oldColor + " " + name
                             + IMAGE_EXTENSION;
            logger.info(PNGDiffCommand.class.getName() + ": Convert String:\n"
                        + cvrtstr);
            process = Runtime.getRuntime().exec(cvrtstr);
        } catch (Exception e1) {
            logger.error(e1.getMessage());
            logger.debug(e1.getMessage(), e1);
        }
        exit = 0; //NOTICEredundant
        if (process != null) {
            try {
                exit = process.waitFor();
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    static public class ImageComponent extends Component {
        BufferedImage img;
        Dimension dim;

        public ImageComponent(BufferedImage image, int dx, int dy) {
            img = image;
            dim = new Dimension(dx, dy);
        }

        @Override
        public Dimension getPreferredSize() {
            return dim;
        }

        public void paint(Graphics g) {
            g.drawImage(img, 0, 0, null);
        }
    }
}