package de.renew.gui;

import de.renew.remote.NetInstanceAccessor;
import de.renew.remote.Server;

import java.awt.Dimension;
import java.awt.List;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import java.rmi.RemoteException;

import java.util.Hashtable;

import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;


/**
 * The net instance selection dialog is the dialog where
 * the user can choose a server net instance to open.
 */
class NetInstanceSelectionFrame extends JFrame {
    public static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                                                       .getLogger(NetInstanceSelectionFrame.class);

    /**
     * The net instance selection frames lookup.
     */
    private static Hashtable<String, NetInstanceSelectionFrame> framesLookup = new Hashtable<String, NetInstanceSelectionFrame>();
    private CPNApplication application;
    private Server server;
    private List netInstanceList;
    private NetInstanceAccessor[] netInstances;
    private String frameID;

    /**
     * Creates a new net instance selection frame.
     * @param application The application this frame is opened for.
     * @param title The title for the frame.
     */
    private NetInstanceSelectionFrame(CPNApplication application, String title) {
        super(title);
        this.application = application;
    }

    /**
     * Brings the net instance selection frame for a given server to the front.
     * If it is not yet open, it is created first.
     * @param application The application this frame is opened for.
     * @param server The server this frame is opened for.
     * @param hostName The host name.
     * @param serverName The server name.
     */
    static synchronized void open(CPNApplication application, Server server,
                                  String hostName, String serverName) {
        String frameID = hostName + ":" + serverName;
        NetInstanceSelectionFrame frame = framesLookup.get(frameID);
        if (frame == null) {
            frame = new NetInstanceSelectionFrame(application,
                                                  "Net instances of "
                                                  + hostName + ":" + serverName);
            frame.frameID = frameID;
            framesLookup.put(frameID, frame);

            frame.server = server;
            frame.setup();
        }
        frame.toFront();
    }

    /**
     * Sets up the frame.
     */
    private void setup() {
        JButton openButton = new JButton("Open");
        openButton.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent event) {
                    try {
                        openDrawing();
                    } catch (RemoteException e) { //NOTICEthrows
                        logger.error(e.getMessage(), e);
                        JOptionPane.showMessageDialog(NetInstanceSelectionFrame.this,
                                                      "A problem occurred: "
                                                      + e + "\n"
                                                      + "See the console for details.",
                                                      "Net instance selection frame",
                                                      JOptionPane.ERROR_MESSAGE);
                    }
                }
            });

        JButton updateButton = new JButton("Refresh");
        updateButton.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent event) {
                    update();
                }
            });

        Box buttonsBox = Box.createHorizontalBox();
        buttonsBox.add(openButton);
        buttonsBox.add(Box.createRigidArea(new Dimension(8, 0)));
        buttonsBox.add(updateButton);

        Box mainBox = Box.createVerticalBox();
        netInstanceList = new List();
        mainBox.add(netInstanceList);
        mainBox.add(Box.createRigidArea(new Dimension(0, 8)));
        mainBox.add(buttonsBox);
        mainBox.add(Box.createRigidArea(new Dimension(0, 8)));

        getContentPane().add(mainBox);
        addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent e) {
                    framesLookup.remove(frameID);
                }
            });

        update();

        setLocation(20, 20);
        setSize(400, 200);
        setVisible(true);
    }

    //NOTICEthrows
    public void openDrawing() throws RemoteException {
        int index = netInstanceList.getSelectedIndex();
        if (index >= 0 && index < netInstances.length) {
            application.openInstanceDrawing(netInstances[index]);
        }
    }

    private void update() {
        try {
            netInstances = server.getSimulator().getNetInstances();


            // copy bindings to array and rebuild the AWT list.
            netInstanceList.removeAll();
            for (int i = 0; i < netInstances.length; i++) {
                netInstanceList.add(netInstances[i].getNet().getName() + ":"
                                    + netInstances[i].getID());
            }
        } catch (RemoteException e) {
            logger.error(e.getMessage(), e);
            JOptionPane.showMessageDialog(this,
                                          "A problem occurred: " + e + "\n"
                                          + "See the console for details.",
                                          "Net instance selection frame",
                                          JOptionPane.ERROR_MESSAGE);
        }
    }
}