package de.renew.gui;

import CH.ifa.draw.DrawPlugin;

import CH.ifa.draw.application.DrawApplication;

import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.framework.FigureEnumeration;
import CH.ifa.draw.framework.FigureWithID;

import CH.ifa.draw.standard.FigureException;

import CH.ifa.draw.util.CommandMenu;
import CH.ifa.draw.util.CommandMenuItem;

import de.renew.application.SimulationEnvironment;
import de.renew.application.SimulatorExtension;

import de.renew.engine.simulator.SimulationThreadPool;

import de.renew.net.Net;
import de.renew.net.NetElementID;
import de.renew.net.NetInstance;
import de.renew.net.Place;
import de.renew.net.PlaceInstance;
import de.renew.net.Transition;
import de.renew.net.TransitionInstance;
import de.renew.net.event.PlaceEventProducer;
import de.renew.net.event.TransitionEventProducer;

import de.renew.remote.NetInstanceAccessor;
import de.renew.remote.RemotePlugin;

import de.renew.shadow.ShadowLookup;

import java.awt.EventQueue;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;

import java.rmi.RemoteException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.swing.JMenu;
import javax.swing.JMenuItem;


/**
 * Manages Breakpoints and keeps all information together needed by
 * <code>Breakpoint</code> objects.
 * <p>
 * Supports two kinds of breakpoints:
 * <ul>
 * <li>Listener: Breakpoint objects listening to transition or place events,
 * hittable within the one net instance to which they are bound or within all
 * instances of a net, depending on the observed net element. These objects are
 * added without modifying the compiled net. Used in any case.</li>
 * <li>Inscriptions: Breakpoint inscriptions added to the net immediately after
 * compilation, which will be hit in every instance of the net. This technique
 * is no longer in use because the listener technique is able to cover this
 * case, too.</li>
 * </ul>
 * </p>
 *
 * BreakpointManager.java Created: Tue May 23 2000
 *
 * @author Michael Duvigneau
 * @see Breakpoint
 */
public class BreakpointManager implements SimulatorExtension {
    private static final int MENU_SHORTCUT_KEY_MASK = Toolkit.getDefaultToolkit()
                                                             .getMenuShortcutKeyMask();
    public static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                                                       .getLogger(BreakpointManager.class);
    private CPNSimulation simulation;

    // private CPNApplication application;
    private CommandMenu breakpointSimulationMenu = null;
    private CommandMenu breakpointNetMenu = null;
    private Vector<Breakpoint> breakpoints = new Vector<Breakpoint>();
    private Vector<BreakpointHitListener> breakpointHitListeners = new Vector<BreakpointHitListener>();
    private Map<Breakpoint, JMenuItem> breakpointMenuItems = new HashMap<Breakpoint, JMenuItem>();

    /**
     * Keeps the hit messages generated by breakpoints to be shown in a single
     * window. This may be wanted if several breakpoints occur in a very short
     * time span. Instead of a single breakpoint hit message all messages will
     * be returned.
     */
    private StringBuffer messageMemory = new StringBuffer();

    public BreakpointManager(CPNSimulation simulation) {
        setSimulation(simulation);
    }

    public void setSimulation(CPNSimulation sim) {
        this.simulation = sim;
    }

    public JMenu getNetMenu() {
        if (breakpointNetMenu == null) {
            breakpointNetMenu = createSetMenu("Breakpoint",
                                              ToggleBreakpointCommand.PRESET);

            breakpointNetMenu.addSeparator();
            breakpointNetMenu.add(new ToggleBreakpointCommand("none", this,
                                                              ToggleBreakpointCommand.REMOVE,
                                                              ToggleBreakpointCommand.PRESET),
                                  KeyEvent.VK_C,
                                  MENU_SHORTCUT_KEY_MASK
                                  + KeyEvent.ALT_DOWN_MASK);

        }
        return breakpointNetMenu;
    }

    public CPNApplication getGui() {
        // this cant be done: the gui might be closed in between
        // if (application == null) {
        // application = (CPNApplication) DrawPlugin.getGui();
        // }
        // return application;
        return (CPNApplication) DrawPlugin.getGui();
    }

    /**
     * Returns the breakpoint menu of this manager. It is created if necessary.
     */
    public JMenu getSimulationMenu() {
        if (breakpointSimulationMenu == null) {
            CommandMenu menu = DrawApplication.createCommandMenu("Breakpoints");

            menu.add(createSetMenus("Set BP at selection"));
            menu.add(createClearMenus("Clear BP at selection"));
            menu.add(new ClearAllBreakpointsCommand("Clear all BPs in current simulation",
                                                    this, simulation),
                     KeyEvent.VK_C,
                     MENU_SHORTCUT_KEY_MASK + KeyEvent.ALT_DOWN_MASK);
            menu.addSeparator();
            breakpointSimulationMenu = menu;
        }
        return breakpointSimulationMenu;
    }

    /**
     * Returns a menu consisting of two setMenus, one for global breakpoints,
     * one for local ones.
     */
    protected CommandMenu createSetMenus(String title) {
        CommandMenu menu = DrawApplication.createCommandMenu(title);

        menu.add(createSetMenu("local", ToggleBreakpointCommand.LOCAL));
        menu.add(createSetMenu("global", ToggleBreakpointCommand.GLOBAL));
        return menu;
    }

    /**
     * Returns a menu with set breakpoint commands respecting the given
     * localGlobal mode, which should be one of the ToggleBreakpointCommand
     * constants LOCAL or GLOBAL. The commands cover all known Breakpoint modes.
     *
     * @see ToggleBreakpointCommand#LOCAL
     * @see ToggleBreakpointCommand#GLOBAL
     */
    protected CommandMenu createSetMenu(String title, int localGlobal) {
        CommandMenu menu = DrawApplication.createCommandMenu(title);

        int bpevent = KeyEvent.VK_B;
        if (localGlobal == ToggleBreakpointCommand.GLOBAL) {
            bpevent = KeyEvent.VK_G;
        }
        menu.add(new ToggleBreakpointCommand("default (t,p)", this,
                                             ToggleBreakpointCommand.ADD,
                                             localGlobal), bpevent,
                 MENU_SHORTCUT_KEY_MASK + KeyEvent.ALT_DOWN_MASK);
        menu.add(new ToggleBreakpointCommand("firing starts (t)", this,
                                             ToggleBreakpointCommand.ADD,
                                             localGlobal, Breakpoint.FIRE));
        menu.add(new ToggleBreakpointCommand("firing completes (t)", this,
                                             ToggleBreakpointCommand.ADD,
                                             localGlobal,
                                             Breakpoint.FIRECOMPLETE));
        menu.add(new ToggleBreakpointCommand("marking changes (p)", this,
                                             ToggleBreakpointCommand.ADD,
                                             localGlobal,
                                             Breakpoint.MARKINGCHANGE));
        menu.add(new ToggleBreakpointCommand("marking changes, ignoring test arcs (p)",
                                             this, ToggleBreakpointCommand.ADD,
                                             localGlobal,
                                             Breakpoint.MARKINGCHANGENOTEST));
        menu.add(new ToggleBreakpointCommand("+ 1 token (p)", this,
                                             ToggleBreakpointCommand.ADD,
                                             localGlobal, Breakpoint.TOKENADDED));
        menu.add(new ToggleBreakpointCommand("- 1 token (p)", this,
                                             ToggleBreakpointCommand.ADD,
                                             localGlobal,
                                             Breakpoint.TOKENREMOVED));
        menu.add(new ToggleBreakpointCommand("test status changes (p)", this,
                                             ToggleBreakpointCommand.ADD,
                                             localGlobal,
                                             Breakpoint.TOKENTESTCHANGE));
        return menu;
    }

    /**
     * Returns a menu consisting of two clearMenus, one for global breakpoints,
     * one for local ones.
     */
    protected CommandMenu createClearMenus(String title) {
        CommandMenu menu = DrawApplication.createCommandMenu(title);

        menu.add(createClearMenu("local", ToggleBreakpointCommand.LOCAL));
        menu.add(createClearMenu("global", ToggleBreakpointCommand.GLOBAL));
        return menu;
    }

    /**
     * Returns a menu with clear breakpoint commands respecting the given
     * localGlobal mode, which should be one of the ToggleBreakpointCommand
     * constants LOCAL or GLOBAL.
     *
     * @see ToggleBreakpointCommand#LOCAL
     * @see ToggleBreakpointCommand#GLOBAL
     */
    protected CommandMenu createClearMenu(String title, int localGlobal) {
        CommandMenu menu = DrawApplication.createCommandMenu(title);

        menu.add(new CommandMenuItem(new ToggleBreakpointCommand("any type",
                                                                 this,
                                                                 ToggleBreakpointCommand.REMOVE,
                                                                 localGlobal)));
        return menu;
    }

    /**
     * Returns a menu with show and clear breakpoint commands concerning the
     * specified breakpoint.
     */
    protected CommandMenu createBPMenu(final Breakpoint bp) {
        CommandMenu menu = DrawApplication.createCommandMenu(bp.toString());

        menu.add(DrawApplication.createMenuItem("show",
                                                new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    showBreakpoint(bp);
                }
            }));
        menu.add(DrawApplication.createMenuItem("clear",
                                                new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    deleteBreakpoint(bp);
                }
            }));
        return menu;
    }

    /**
     * Creates one breakpoint at the given transition instance.
     *
     * The created breakpoint is added to the internal list of known breakpoints
     * automatically.
     *
     * @param instance
     *            the transition instance to observe
     * @param mode
     *            on which event to react on, given as <code>Breakpoint</code>
     *            mode.
     *
     * @return null if the mode was not valid, a breakpoint object otherwise.
     *
     * @see #isValidTransitionMode
     */
    public Breakpoint createTransitionInstanceBreakpoint(TransitionInstance instance,
                                                         int mode) {
        return createTransitionListenerBreakpoint(instance, mode,
                                                  instance.getNetInstance()
                                                          .getNet());
    }

    /**
     * Creates one breakpoint at the given transition which will be hit at any
     * instance of that transition.
     *
     * The created breakpoint is added to the internal list of known breakpoints
     * automatically.
     *
     * @param transition
     *            the transition to observe
     * @param mode
     *            on which event to react on, given as <code>Breakpoint</code>
     *            mode.
     * @param net
     *            the net the transition belongs to
     *
     * @return null if the mode was not valid, a breakpoint object otherwise.
     *
     * @see #isValidTransitionMode
     */
    public Breakpoint createTransitionBreakpoint(Transition transition,
                                                 int mode, Net net) {
        return createTransitionListenerBreakpoint(transition, mode, net);
    }

    /**
     * The common method body of
     *
     * @see #createTransitionBreakpoint
     * @see #createTransitionInstanceBreakpoint
     */
    private Breakpoint createTransitionListenerBreakpoint(TransitionEventProducer producer,
                                                          int mode, Net net) {
        if (!isValidTransitionMode(mode)) {
            return null;

        }
        TransitionInstanceBreakpoint newBreakpoint = new TransitionInstanceBreakpoint(this,
                                                                                      producer,
                                                                                      mode,
                                                                                      net);

        addBreakpoint(newBreakpoint);
        return newBreakpoint;
    }

    /**
     * Creates one breakpoint at the given place instance.
     *
     * The created breakpoint is added to the internal list of known breakpoints
     * automatically.
     *
     * @param instance
     *            the place instance to observe
     * @param mode
     *            on which event to react on, given as <code>Breakpoint</code>
     *            mode.
     *
     * @return null if the mode was not valid, a breakpoint object otherwise.
     *
     * @see #isValidPlaceMode
     */
    public Breakpoint createPlaceInstanceBreakpoint(PlaceInstance instance,
                                                    int mode) {
        return createPlaceListenerBreakpoint(instance, mode,
                                             instance.getNetInstance().getNet());
    }

    /**
     * Creates one breakpoint at the given place, which will be hit at any
     * instance of that place.
     *
     * The created breakpoint is added to the internal list of known breakpoints
     * automatically.
     *
     * @param place
     *            the place to observe
     * @param mode
     *            on which event to react on, given as <code>Breakpoint</code>
     *            mode.
     * @param net
     *            the net the the place belongs to
     *
     * @return null if the mode was not valid, a breakpoint object otherwise.
     *
     * @see #isValidPlaceMode
     */
    public Breakpoint createPlaceBreakpoint(Place place, int mode, Net net) {
        return createPlaceListenerBreakpoint(place, mode, net);
    }

    /**
     * The common method body of
     *
     * @see #createPlaceBreakpoint
     * @see #createPlaceInstanceBreakpoint
     */
    private Breakpoint createPlaceListenerBreakpoint(PlaceEventProducer producer,
                                                     int mode, Net net) {
        if (!isValidPlaceMode(mode)) {
            return null;

        }
        PlaceInstanceBreakpoint newBreakpoint = new PlaceInstanceBreakpoint(this,
                                                                            producer,
                                                                            mode,
                                                                            net);

        addBreakpoint(newBreakpoint);
        return newBreakpoint;
    }

    /**
     * Creates one global breakpoint at the given transition. The created
     * breakpoint is added to the internal list of known breakpoints
     * automatically.
     * <p>
     * <b>Caution:</b> As there will some inscriptions be removed from the net,
     * the simulation must not run while this method executes! The same
     * condition must be met when the breakpoint is to be deleted.
     * </p>
     *
     * @param transition
     *            the transition to observe
     * @param mode
     *            on which events to react on, given as <code>Breakpoint</code>
     *            modes. Valid values are: DEFAULT, INSCRIPTION.
     * @param net
     *            the net to which the transition belongs
     *
     * @return <code>null</code> if the mode was not valid, a breakpoint object
     *         otherwise.
     *
     * @see Breakpoint#DEFAULT
     * @see Breakpoint#INSCRIPTION
     *
     * @deprecated Use <code>createTransitionBreakpoint(transition,
     * mode, net)</code> instead.
     * @see #createTransitionBreakpoint
     */
    public GlobalTransitionBreakpoint createGlobalTransitionBreakpoint(Transition transition,
                                                                       int mode,
                                                                       Net net) {
        if ((mode != Breakpoint.INSCRIPTION) & (mode != Breakpoint.DEFAULT)) {
            return null;

        }
        GlobalTransitionBreakpoint newBreakpoint = new GlobalTransitionBreakpoint(this,
                                                                                  transition,
                                                                                  net);

        addBreakpoint(newBreakpoint);
        return newBreakpoint;
    }

    /**
     * Returns a list of all {@link Breakpoint}s set for the given
     * {@link NetElement}.
     *
     * @see #isBreakpointSetAt(Object)
     * @param netElement
     *            the net element
     * @return the list of breakpoints
     */
    public List<Breakpoint> getBreakpointsAt(Object netElement) {
        List<Breakpoint> result = new ArrayList<Breakpoint>();
        for (Breakpoint bp : breakpoints) {
            if (bp.getTaggedElement() == netElement) {
                result.add(bp);
            }
        }
        return result;
    }

    /**
     * Checks whether a {@link Breakpoint}s has been set for the given
     * {@link NetElement}.
     *
     * @see #getBreakpointsAt(Object)
     * @param netElement
     *            the net element
     * @return true if there is at least one breakpoint set, false otherwise
     */
    public boolean isBreakpointSetAt(Object netElement) {
        // Instead of returning getBreakpointsAt(netElement).size() > 0, we
        // iterate the list to save the overhead of constructing a list.
        for (Breakpoint bp : breakpoints) {
            if (bp.getTaggedElement() == netElement) {
                return true;
            }
        }
        return false;
    }

    /**
     * Deletes all breakpoints attached to the given place, transition or place
     * or transition instance.
     * <p>
     * Will not cross the global/local distinction: If a transition is given,
     * breakpoints attached to local instances of that transition will not be
     * deleted. Naturally, this holds also the other way around.
     * </p>
     *
     * @param netElement
     *            the place, transition or place or transition instance which
     *            should be freed of breakpoints.
     * @return number of deleted breakpoints
     */
    public int deleteBreakpointsAt(Object netElement) {
        List<Breakpoint> points = getBreakpointsAt(netElement);
        for (Breakpoint bp : points) {
            deleteBreakpoint(bp);
        }
        return points.size();
    }

    /**
     * Clears all currently known breakpoints. This method should be called at
     * least when the current simulation is terminated.
     */
    public void deleteAllBreakpoints() {
        // As the deleteBreakpoint call will always call
        // breakpoints.remove(...) we would run into a
        // ConcurrentModificationException when using an
        // Iterator to find and remove all breakpoints.
        // Thus we just keep deleting the last breakpoint
        // until no breakpoints are left.
        int size;
        while ((size = breakpoints.size()) > 0) {
            deleteBreakpoint(breakpoints.get(size - 1));
        }
    }

    /**
     * Clears the specified breakpoint. Releases its resources and removes it
     * from the list of known breakpoints.
     */
    public void deleteBreakpoint(Breakpoint bp) {
        bp.release();
        removeBreakpoint(bp);
    }

    /**
     * Attaches global breakpoints to all places or transitions with breakpoint
     * attribute at the corresponding graphical figure. Traverses the drawings
     * of all nets included in the given shadow lookup, if they are opened.
     * Takes care not to request drawings from the drawing loader.
     * <p>
     * There is no check if any of these breakpoints already exist. So this
     * method should be called only once per simulation.
     * </p>
     */
    protected void addPresetBreakpoints(ShadowLookup lookup) {
        String currentNetName;
        CPNDrawing currentDrawing;
        FigureEnumeration figures;
        Figure currentFigure;
        Integer currentMode;
        TransitionFigure currentTransFig;
        Transition currentTransition;
        PlaceFigure currentPlaceFig;
        Place currentPlace;
        Net currentNet;
        Breakpoint bp;
        CPNDrawingLoader drawingLoader = ModeReplacement.getInstance()
                                                        .getDrawingLoader();

        for (Iterator<String> it = lookup.allNewlyCompiledNetNames();
                     it.hasNext();) {
            currentNetName = it.next();
            currentNet = lookup.getNet(currentNetName);

            // To stay with the old behaviour, we deactivate the
            // drawing loading mechanism when looking for drawings.
            // This way only breakpoints for nets with open
            // drawings will be set, no new drawings will be opened.
            // TODO: This behaviour should be switchable.
            currentDrawing = drawingLoader.getDrawing(currentNetName, false);
            if (currentDrawing == null) {
                logger.debug("BreakpointManager: Could not add preset breakpoints for net "
                             + currentNetName + ": no drawing found.");
            } else {
                if (logger.isDebugEnabled()) {
                    logger.debug(BreakpointManager.class.getName()
                                 + ": Looking for breakpoints in "
                                 + currentNetName);
                }
                figures = currentDrawing.figures();
                while (figures.hasMoreElements()) {
                    currentFigure = figures.nextFigure();
                    currentMode = (Integer) currentFigure.getAttribute(Breakpoint.ATTRIBUTENAME);
                    if (currentMode != null) {
                        if (currentFigure instanceof TransitionFigure) {
                            currentTransFig = (TransitionFigure) currentFigure;
                            int id = currentTransFig.getID();
                            currentTransition = currentNet.getTransitionWithID(new NetElementID(id));
                            // in most cases this simple NetElementID should
                            // suffice
                            if (currentTransition == null) {
                                logger.warn(BreakpointManager.class.getName()
                                            + ": Found no Transition for Breakpoint on Figure "
                                            + currentTransFig);
                            } else {
                                bp = createTransitionBreakpoint(currentTransition,
                                                                currentMode
                                         .intValue(), currentNet);
                                logger.debug("Created Breapoint: " + bp);
                            }
                        } else if (currentFigure instanceof PlaceFigure) {
                            currentPlaceFig = (PlaceFigure) currentFigure;
                            int id = currentPlaceFig.getID();
                            currentPlace = currentNet.getPlaceWithID(new NetElementID(id));
                            // in most cases this simple NetElementID should
                            // suffice
                            if (currentPlace == null) {
                                logger.warn(BreakpointManager.class.getName()
                                            + ": Found no Place for Breakpoint on Figure "
                                            + currentPlaceFig);
                            } else {
                                bp = createPlaceBreakpoint(currentPlace,
                                                           currentMode.intValue(),
                                                           currentNet);
                                if (logger.isDebugEnabled()) {
                                    logger.debug("Created Breapoint: " + bp);
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * Appends the given breakpoint to the end of the list of known breakpoints.
     * Keeps the menu consistent.
     */
    private void addBreakpoint(Breakpoint newBreakpoint) {
        breakpoints.add(newBreakpoint);
        JMenuItem mi = breakpointSimulationMenu.add(createBPMenu(newBreakpoint));
        breakpointMenuItems.put(newBreakpoint, mi);
    }

    /**
     * Removes the given breakpoint from the list of known breakpoints. Keeps
     * the menu consistent.
     */
    private void removeBreakpoint(Breakpoint breakpoint) {
        breakpoints.remove(breakpoint);
        JMenuItem mi = breakpointMenuItems.remove(breakpoint);
        breakpointSimulationMenu.remove(mi);
    }

    /**
     * Stops the running simulation (which is what breakpoints should do) by
     * using a concurrent thread.
     * <p>
     * The extra thread is needed to avoid deadlock situations within
     * <code>de.renew.engine.simulator.Simulator.stopRun()</code>
     * implementations. The deadlock occurs because the <code>stopRun()</code>
     * methods try to ensure that the simulation has stopped, but the execution
     * of this breakpoint is part of the running simulation - and it would stop
     * only after the <code>stopRun()</code> waiting condition is met...
     * </p>
     * <p>
     * <b>Drawback</b>: This method does not guarantee that the simulation is
     * stopped after the method has finished.
     * </p>
     */
    protected void stopSimulation() {
        SimulationThreadPool.getCurrent().execute(new Runnable() {
                public void run() {
                    simulation.simulationStop();
                }
            });
    }

    /**
     * Informs the BreakpointManager that the simulation hits a breakpoint.
     *
     * @param breakpoint
     *            the hit breakpoint
     */
    protected void informHitBreakpoint(final Breakpoint breakpoint) {
        // sent BreakpointHitEvent to all registered BreakpointHitListeners
        BreakpointHitEvent event = new BreakpointHitEvent(breakpoint);

        BreakpointHitListener[] listeners = this.breakpointHitListeners.toArray(new BreakpointHitListener[] {  });
        for (int x = 0; x < listeners.length; x++) {
            listeners[x].hitBreakpoint(event);
        }

        if (!(event.isConsumed())) {
            showHitBreakpoint(breakpoint);
        }
    }

    /**
     * Shows a message dialog telling that the given breakpoint was hit and
     * selects its associated net element figure. Used by breakpoints when they
     * occur.
     * <p>
     * This method detaches from the current thread and synchronises its body
     * with the AWT event queue. This behaviour is needed because this method is
     * mostly called from simulation threads which should not wait on AWT-only
     * locks.
     * </p>
     *
     * @param breakpoint
     *            the breakpoint which has occured.
     */
    protected void showHitBreakpoint(final Breakpoint breakpoint) {
        EventQueue.invokeLater(new Runnable() {
                public void run() {
                    DrawPlugin.getGui()
                              .showStatus("A breakpoint was hit: " + breakpoint
                                          + ".");
                    FigureException e = locateBreakpoint(breakpoint, true);

                    if (e != null) {
                        GuiPlugin.getCurrent().processFigureException(e, true);
                    }
                }
            });
    }

    /**
     * Selects the associated net element figure of the given Breakpoint. Used
     * by breakpoint menu entries of kind "Show breakpoint".
     *
     * @param breakpoint
     *            the breakpoint to show.
     */
    protected void showBreakpoint(Breakpoint breakpoint) {
        assert EventQueue.isDispatchThread() : "showBreakpoint must be called within AWT event queue.";
        DrawPlugin.getGui().showStatus("Selecting " + breakpoint + ".");
        FigureException e = locateBreakpoint(breakpoint, false);

        if (e != null) {
            getGui().selectOffendingElements(e);
        }
    }

    /**
     * Adds a BreakpointHitListener
     *
     * @param listener
     *            the BreakpointHitListener to add
     */
    public void addBreakpointHitListener(BreakpointHitListener listener) {
        this.breakpointHitListeners.add(listener);
    }

    /**
     * Removes a registered BreakpointHitListener.
     *
     * @param listener
     *            the BreakpointHitListener to remove
     */
    public void removeBreakpointHitListener(BreakpointHitListener listener) {
        this.breakpointHitListeners.remove(listener);
    }

    /**
     * Empties the memory of generated beakpoint hit messages so they won't be
     * shown again with the next hit breakpoint.
     */
    public void clearLog() {
        messageMemory = new StringBuffer();
    }

    /**
     * Looks up the associated net element figure of the given breakpoint and
     * returns it in a <code>FigureException</code>. The hit element of the
     * breakpoint will be used, if <code>hit
     * </code>is <code>true</code>, else the tagged element will be looked up.
     * <p>
     * The FigureException will also include a generated hit message, if
     * requested. If the mapping to a Figure was not possible, the messages are
     * printed to System.err.
     * </p>
     *
     * @param breakpoint
     *            the breakpoint to look up.
     * @param hit
     *            if true, an hit message is generated.
     * @return the figure exception containing the message, the element figure
     *         and the drawing; <code>null</code> if no figure was found.
     *
     * @see #showHitBreakpoint
     * @see #showBreakpoint
     */
    private FigureException locateBreakpoint(Breakpoint breakpoint, boolean hit) {
        String message = "";
        String allMessages;
        Net net = null;
        NetInstance netInstance = null;
        int id = FigureWithID.NOID;
        Object netElement = hit ? breakpoint.getHitElement()
                                : breakpoint.getTaggedElement();

        if (netElement instanceof PlaceInstance) {
            PlaceInstance placeInst = (PlaceInstance) netElement;

            id = placeInst.getPlace().getID().getFigureID();
            netInstance = placeInst.getNetInstance();
            net = netInstance.getNet();
            if (hit) {
                message = "Hit " + breakpoint + ".";
            }
        } else if (netElement instanceof TransitionInstance) {
            TransitionInstance transitionInst = (TransitionInstance) netElement;

            id = transitionInst.getTransition().getID().getFigureID();
            netInstance = transitionInst.getNetInstance();
            net = netInstance.getNet();
            if (hit) {
                message = "Hit " + breakpoint + ".";
            }
        } else if (netElement instanceof Transition) {
            Transition transition = (Transition) netElement;

            id = transition.getID().getFigureID();
            net = breakpoint.getTaggedNet();

            // No hit message as static transitions cannot fire.
        } else if (netElement instanceof Place) {
            Place place = (Place) netElement;

            id = place.getID().getFigureID();
            net = breakpoint.getTaggedNet();

            // No hit message as static places cannot change anything.
        } else {
            logger.error("Cannot determine the location of breakpoint at "
                         + netElement + "!?");
            return null;
        }

        if (hit) {
            allMessages = messageMemory.toString() + "\n" + message;
            messageMemory.append("(");
            messageMemory.append(message);
            messageMemory.append(")\n");
        } else {
            allMessages = message;
        }

        String netName = net.getName();
        CPNDrawing drawing = ModeReplacement.getInstance().getDrawingLoader()
                                            .getDrawing(netName);

        if (drawing == null) {
            message = message
                      + "\nSorry, cannot show the location of breakpoint at "
                      + netElement + " because net " + netName
                      + " is not loaded.";
            getGui().showStatus(message);
            logger.error(message);
            return null;
        } else if (netInstance == null) {
            FigureWithID figure = drawing.getFigureWithID(id);

            return new FigureException("Renew: Breakpoint", allMessages,
                                       drawing, figure);
        } else {
            try {
                if (RemotePlugin.getInstance() != null) {
                    NetInstanceAccessor instanceAcc = RemotePlugin.getInstance()
                                                                  .wrapInstance(netInstance);
                    getGui().openInstanceDrawing(instanceAcc);
                    CPNInstanceDrawing instanceDrawing = CPNInstanceDrawing
                                                             .getInstanceDrawing(instanceAcc);
                    FigureWithID figure = drawing.getFigureWithID(id);
                    InstanceFigure instanceFigure = instanceDrawing
                                                        .getInstanceFigure(figure);

                    return new FigureException("Renew: Breakpoint",
                                               allMessages, instanceDrawing,
                                               instanceFigure);
                } else {
                    return null;
                }
            } catch (RemoteException e) {
                logger.error(e.getMessage(), e);
                return null;
            }
        }
    }

    /* Non-JavaDoc: Specified by the SimulatorExtension interface. */
    public void simulationSetup(SimulationEnvironment env) {
    }

    /* Non-JavaDoc: Specified by the SimulatorExtension interface. */
    public void netsCompiled(ShadowLookup lookup) {
        if (lookup.containsNewlyCompiledNets()) {
            addPresetBreakpoints(lookup);
        }
        clearLog();
    }

    /* Non-JavaDoc: Specified by the SimulatorExtension interface. */
    public void simulationTerminated() {
        deleteAllBreakpoints();
        clearLog();
    }

    public void simulationTerminating() {
        // Nothing to do ?
    }

    /**
     * Tells wether the given mode is valid for transition breakpoints.
     * <p>
     * Valid values are: DEFAULT, FIRE, FIRECOMPLETE.
     * </p>
     *
     * @return <code>true</code>, if the given breakpoint mode is applicable to
     *         transitions or transition instances. <br>
     *         <code>false</code>, otherwise.
     *
     * @see Breakpoint#DEFAULT
     * @see Breakpoint#FIRE
     * @see Breakpoint#FIRECOMPLETE
     */
    public static boolean isValidTransitionMode(int mode) {
        return (mode == Breakpoint.FIRE) || (mode == Breakpoint.FIRECOMPLETE)
               || (mode == Breakpoint.DEFAULT);
    }

    /**
     * Tells wether the given mode is valid for place breakpoints.
     * <p>
     * Valid values are: DEFAULT, MARKINGCHANGE, MARKINGCHANGENOTEST,
     * TOKENADDED, TOKENREMOVED, TOKENTESTCHANGE.
     * </p>
     *
     * @return <code>true</code>, if the given breakpoint mode is applicable to
     *         places or place instances. <br>
     *         <code>false</code>, otherwise.
     *
     * @see Breakpoint#DEFAULT
     * @see Breakpoint#MARKINGCHANGE
     * @see Breakpoint#MARKINGCHANGENOTEST
     * @see Breakpoint#TOKENADDED
     * @see Breakpoint#TOKENREMOVED
     * @see Breakpoint#TOKENTESTCHANGE
     */
    public static boolean isValidPlaceMode(int mode) {
        return (mode == Breakpoint.MARKINGCHANGE)
               || (mode == Breakpoint.MARKINGCHANGENOTEST)
               || (mode == Breakpoint.TOKENADDED)
               || (mode == Breakpoint.TOKENREMOVED)
               || (mode == Breakpoint.TOKENTESTCHANGE)
               || (mode == Breakpoint.DEFAULT);
    }
    //
    // public class BreakpointSimulationMenuExtender extends MenuExtenderAdapter
    // {
    // public BreakpointSimulationMenuExtender() {
    // super(GuiPlugin.SIMULATION_MENU,
    // new JMenuItem[] { getSimulationMenu() });
    // }
    // }
    //
    // public class BreakpointNetMenuExtender extends MenuExtenderAdapter {
    // public BreakpointNetMenuExtender() {
    // super(GuiPlugin.NET_MENU, new JMenuItem[] { getNetMenu() });
    // }
    // }
}