/*
 * @(#)Iconkit.java 5.1
 *
 */
package CH.ifa.draw.util;

import java.awt.Component;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.Toolkit;
import java.awt.image.ImageProducer;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;


/**
 * The Iconkit class supports the sharing of images. It maintains
 * a map of image names and their corresponding images.
 *
 * Iconkit also supports to load a collection of images in
 * synchronized way.
 * The resolution of a path name to an image is delegated to the DrawingEditor.
 * <hr>
 * <b>Design Patterns</b><P>
 * <img src="images/red-ball-small.gif" width=6 height=6 alt=" o ">
 * <b><a href=../pattlets/sld031.htm>Singleton</a></b><br>
 * The Iconkit is a singleton.
 * <hr>
 */
public class Iconkit {
    public static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                                                       .getLogger(Iconkit.class);
    private final static int ID = 123;
    private static Iconkit fgIconkit = null;
    private Hashtable<String, Image> fMap;
    private Vector<String> fRegisteredImages;
    private Component fComponent;

    /**
     * Constructs an Iconkit that uses the given editor to
     * resolve image path names.
     */
    public Iconkit(Component component) {
        fMap = new Hashtable<String, Image>(53);
        fRegisteredImages = new Vector<String>(10);
        fComponent = component;
        fgIconkit = this;
    }

    /**
     * Gets the single instance
     */
    public static Iconkit instance() {
        return fgIconkit;
    }

    /**
     * Loads all registered images.
     * @see #registerImage
     */
    public void loadRegisteredImages(Component component) {
        if (fRegisteredImages.size() == 0) {
            return;
        }

        MediaTracker tracker = new MediaTracker(component);


        // register images with MediaTracker
        Enumeration<String> k = fRegisteredImages.elements();
        while (k.hasMoreElements()) {
            String fileName = k.nextElement();
            if (basicGetImage(fileName) == null) {
                tracker.addImage(loadImage(fileName), ID);
            }
        }
        fRegisteredImages.removeAllElements();

        // block until all images are loaded
        try {
            tracker.waitForAll();
        } catch (Exception e) {
        }
    }

    /**
     * Registers an image that is then loaded together with
     * the other registered images by loadRegisteredImages.
     * @see #loadRegisteredImages
     */
    public void registerImage(String fileName) {
        fRegisteredImages.addElement(fileName);
    }

    /**
     * Registers and loads an image.
     */
    public Image registerAndLoadImage(Component component, String fileName) {
        registerImage(fileName);
        loadRegisteredImages(component);
        return getImage(fileName);
    }

    /**
     * Loads an image with the given name.
     */
    public Image loadImage(String filename) {
        if (fMap.containsKey(filename)) {
            return fMap.get(filename);
        }
        Image image = loadImageResource(filename);
        if (image != null) {
            fMap.put(filename, image);
        }
        return image;
    }

    public Image loadImageResource(String resourcename) {
        Toolkit toolkit = Toolkit.getDefaultToolkit();
        try {
            java.net.URL url = getClass().getResource(resourcename);
            logger.debug(resourcename + "(" + url + ")");
            if (url == null) {
                // if url is null: use ressourcename!
                return toolkit.getImage(resourcename);
            }
            Object content = url.getContent();
            if (content instanceof ImageProducer) {
                return toolkit.createImage((ImageProducer) content);
            } else if (content instanceof java.io.InputStream) {
                java.io.InputStream stream = (java.io.InputStream) content;
                int max = 1024;
                if (stream.available() > max) {
                    max = stream.available();
                }
                byte[] data = new byte[max];
                int pos = 0;
                boolean canReadMore;
                do {
                    if (pos > max / 2) {
                        max = max * 2;
                        byte[] newData = new byte[max];
                        System.arraycopy(data, 0, newData, 0, pos);
                        data = newData;
                    }
                    int num = stream.read(data, pos, max - pos);
                    canReadMore = num > 0;
                    if (canReadMore) {
                        pos = pos + num;
                    }
                } while (canReadMore);
                ((java.io.InputStream) content).close();

                // Create the image.
                return toolkit.createImage(data, 0, pos);
            } else {
                // Since we have no useful information available,
                // we try to go the more resource intensive route.
                return toolkit.getImage(url);
            }
        } catch (Exception ex) {
            logger.error("While loading " + resourcename + ":");
            logger.error(ex.getMessage(), ex);
            return null;
        }
    }

    /**
     * Gets the image with the given name. If the image
     * can't be found it tries it again after loading
     * all the registered images.
     */
    public Image getImage(String filename) {
        Image image = basicGetImage(filename);
        if (image != null) {
            return image;
        }


        // load registered images and try again
        loadRegisteredImages(fComponent);
        // try again
        return basicGetImage(filename);
    }

    private Image basicGetImage(String filename) {
        if (fMap.containsKey(filename)) {
            return fMap.get(filename);
        }
        return null;
    }
}