package CH.ifa.draw.util;

import java.awt.EventQueue;


/**
 * The class is a <code>Runnable</code> that schedules itself for
 * execution in the AWT event queue when appropriate. In parallel, each
 * <code>UpdateTask</code> instance acts as a synchronization point
 * where new update notifications are merged with an unfinished update
 * request that has been scheduled before.
 * </p>
 * @author Michael Duvigneau
 * @version 1.0
 * @since Renew 2.1
 **/
public class AWTSynchronizedUpdate implements Runnable {
    private boolean inUpdate = false;
    private boolean anotherTurn = false;
    private Runnable updateTask;

    /**
     * Creates a new <code>AWTSynchronizedUpdate</code> instance that
     * schedules the given <code>updateTask</code> when requested.
     *
     * @param updateTask  a <code>Runnable</code> that execute the real
     *                    work in the context of the AWT event queue.
     **/
    public AWTSynchronizedUpdate(Runnable updateTask) {
        this.updateTask = updateTask;
    }

    /**
     * Schedules the task with the AWT event queue if there is not
     * already a pending update event. Otherwise the scheduled task is
     * informed that it should recheck its results.
     **/
    public synchronized void scheduleUpdate() {
        anotherTurn = true;
        if (!inUpdate) {
            EventQueue.invokeLater(this);
            inUpdate = true;
        }
    }

    /**
     * Determines whether there is a pending update request and updates
     * the flags for pending events according to the result.
     * <p>
     * This is a test-and-set method with multiple side effects:
     * <ul>
     * <li> If there are no pending updates (flag <code>anotherTurn</code>
     *      is <code>false</code>), the flag <code>inUpdate</code> is
     *      set to <code>false</code> to indicate that all pending
     *      updates have been handled.
     * </li>
     * <li> In any case, the flag <code>anotherTurn</code> is reset to
     *      <code>false</code>.
     * </li>
     * </ul>
     * </p>
     *
     * @return the former value of the flag <code>anotherTurn</code>.
     **/
    private synchronized boolean testAndResetUpdateRequest() {
        inUpdate = anotherTurn;
        anotherTurn = false;
        return inUpdate;
    }

    /**
     * Updates the associated <code>PlaceInstanceUpdateTask</code> as long
     * as update notifications arrive and the updateTask has not been
     * released.
     **/
    public void run() {
        while (testAndResetUpdateRequest()) {
            updateTask.run();
        }
    }
}