/*
 * @(#)SelectionTool.java 5.1
 *
 */
package CH.ifa.draw.standard;

import CH.ifa.draw.figures.ClickableDecoration;

import CH.ifa.draw.framework.ChildFigure;
import CH.ifa.draw.framework.Decoration;
import CH.ifa.draw.framework.DrawingEditor;
import CH.ifa.draw.framework.DrawingView;
import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.framework.Handle;
import CH.ifa.draw.framework.Tool;

import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.InputEvent;
import java.awt.event.MouseEvent;


/**
 * Tool to select and manipulate figures.
 * A selection tool is in one of three states, e.g., background
 * selection, figure selection, handle manipulation. The different
 * states are handled by different child tools.
 * <hr>
 * <b>Design Patterns</b><P>
 * <img src="images/red-ball-small.gif" width=6 height=6 alt=" o ">
 * <b><a href=../pattlets/sld032.htm>State</a></b><br>
 * SelectionTool is the StateContext and child is the State.
 * The SelectionTool delegates state specific
 * behavior to its current child tool.
 * <hr>
 */
public class SelectionTool extends AbstractTool {
    public static org.apache.log4j.Logger logger = org.apache.log4j.Logger
                                                       .getLogger(SelectionTool.class);
    private Tool fChild = null;
    private Tool fLastChild = null;
    private DrawingView fFreezedView = null;

    public SelectionTool(DrawingEditor editor) {
        super(editor);
    }

    /**
     * Handles mouse down events and starts the corresponding tracker.
     *
     * Caution: This method freezes the view until a
     * <code>mouseReleased</code> event is received or the tool is
     * <code>deactivate</code>d.
     */
    public void mouseDown(MouseEvent e, int x, int y) {
        // on Windows NT: AWT generates additional mouse down events
        // when the left button is down && right button is clicked.
        // To avoid dead locks we ignore such events
        if (fChild != null) {
            return;
        }


        // Since Java 1.4.2, the AWT event thread dies after an Error or
        // RuntimeExeption. Because the mouseUp event then belongs to a
        // different thread, it cannot unfreeze the view. So, we have to
        // catch any abnormal situation and unfreeze the view immediately.
        try {
            boolean rightclick = (e.getModifiers()
                                 & (InputEvent.BUTTON2_MASK
                                   | InputEvent.BUTTON3_MASK)) != 0;
            if (fLastChild != null && e.getClickCount() == 2 && !rightclick) {
                // If left-double-click: use previous fChild!
                fChild = fLastChild;
                fLastChild = null;
            } else {
                if (fFreezedView == null) {
                    fFreezedView = view();
                    fFreezedView.freezeView();
                }

                Handle handle = null;
                if (!rightclick && view().selectionCount() == 1) {
                    handle = view().findHandle(e.getX(), e.getY());
                }
                if (handle != null) {
                    fChild = createHandleTracker(fEditor, handle);
                    fLastChild = fChild; // store for double-clicks!
                } else { // don't clicked handle
                    fLastChild = null;
                    Figure figure = drawing().findFigure(e.getX(), e.getY());


                    // also interprete clicks on a figure's handle as clicks
                    // on the figure:
                    if (figure == null) {
                        handle = view().findHandle(e.getX(), e.getY());
                        if (handle != null) {
                            figure = handle.owner();
                        }
                    }
                    if (figure != null) {
                        if (logger.isDebugEnabled()) {
                            logger.debug("Click-on-figure: " + figure);
                        }
                        if (e.getClickCount() >= 2 && (!rightclick)) {
                            // if (rightclick && logger.isDebugEnabled()) {
                            //    logger.debug("This figure "+figure);
                            // } else {
                            figure.inspect(view(), false);
                            // }
                            return;
                        } else if (rightclick) {
                            // logger.debug("Right-click!");
                            if (alternateInspectFigure(figure)) {
                                return;
                            }
                        } else if (figure instanceof AbstractFigure
                                           && ((AbstractFigure) figure)
                                               .isDecorated()) { // decorated figure
                            Decoration deco = drawing()
                                                  .findDecoration(e.getX(),
                                                                  e.getY());
                            if (deco != null
                                        && (deco instanceof ClickableDecoration)) { // click on a Decoration
                                ((ClickableDecoration) deco).clicked();
                                return;
                            }
                        }
                        fChild = createDragTracker(fEditor, figure);
                    } else { // figure == null
                        Decoration deco = drawing()
                                              .findDecoration(e.getX(), e.getY());
                        if (deco != null
                                    && (deco instanceof ClickableDecoration)) { // click on a Decoration
                            ((ClickableDecoration) deco).clicked();
                        }
                        if (!e.isShiftDown()) {
                            view().clearSelection();
                        }
                        fChild = createAreaTracker(fEditor);
                    }
                }
            }
            fChild.mouseDown(e, x, y);
        } catch (RuntimeException ex) {
            if (fFreezedView != null) {
                fFreezedView.unfreezeView();
                fFreezedView = null;
            }
            throw ex;
        } catch (Error ex) {
            if (fFreezedView != null) {
                fFreezedView.unfreezeView();
                fFreezedView = null;
            }
            throw ex;
        }
    }

    /**
     * Handles mouse drag events. The events are forwarded to the
     * current tracker.
     */
    public void mouseDrag(MouseEvent e, int x, int y) {
        if (fChild != null) { // JDK1.1 doesn't guarantee mouseDown, mouseDrag, mouseUp
            fChild.mouseDrag(e, x, y);
            fLastChild = null; // don't do double-clicks after dragging
        }
    }

    /**
     * Handles mouse up events. The events are forwarded to the
     * current tracker.
     * <p>
     * Unfreezes the view, if frozen by a <code>mouseDown</code> event.
     * </p>
     */
    public void mouseUp(MouseEvent e, int x, int y) {
        if (fFreezedView != null) {
            fFreezedView.unfreezeView();
            fFreezedView = null;
        }
        if (fChild != null) { // JDK1.1 doesn't guarantee mouseDown, mouseDrag, mouseUp
            fChild.mouseUp(e, x, y);
        }
        fChild = null;
    }

    /**
     * Factory method to create a Handle tracker. It is used to track a handle.
     */
    protected Tool createHandleTracker(DrawingEditor editor, Handle handle) {
        return new HandleTracker(editor, handle);
    }

    /**
     * Factory method to create a Drag tracker. It is used to drag a figure.
     */
    protected Tool createDragTracker(DrawingEditor editor, Figure f) {
        if (f instanceof ChildFigure) {
            return new ChildDragTracker(editor, (ChildFigure) f);
        } else {
            return new DragTracker(editor, f);
        }
    }

    /**
     * Factory method to create an area tracker. It is used to select an
     * area.
     */
    protected Tool createAreaTracker(DrawingEditor editor) {
        return new SelectAreaTracker(editor);
    }

    protected boolean alternateInspectFigure(Figure f) {
        return f.inspect(view(), true);
    }

    /**
     * Delegates the graphical feedback update to the child.
     **/
    public void draw(Graphics g) {
        if (fChild != null) {
            fChild.draw(g);
        }
    }

    /**
     * {@inheritDoc}
     * <p>
     * Unfreezes the view, if frozen by a <code>mouseDown</code> event.
     * </p>
     **/
    public void deactivate() {
        if (fFreezedView != null) {
            fFreezedView.unfreezeView();
            fFreezedView = null;
        }
        super.deactivate();
    }
}