/*
 * @(#)ChangeAttributeCommand.java 5.1
 *
 */
package CH.ifa.draw.standard;

import CH.ifa.draw.DrawPlugin;

import CH.ifa.draw.application.MenuManager;

import CH.ifa.draw.framework.DrawingEditor;
import CH.ifa.draw.framework.DrawingView;
import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.framework.FigureEnumeration;

import CH.ifa.draw.util.Command;

import java.awt.Button;
import java.awt.Color;
import java.awt.Dialog;
import java.awt.FlowLayout;
import java.awt.Panel;
import java.awt.TextField;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.JFrame;


/**
 * Command to change a named figure attribute.
 * <p>
 * The command's effects are undoable step by step.
 * Each use of the apply button can be undone separately.
 * So this command doesn't need to inherit UndoableCommand.
 * </p>
 */
public class QueryAttributeCommand extends Command {
    // DrawingEditor fEditor;
    // Dialog dialog;
    String fAttribute;
    TextField input;
    private Class<?> type;
    String displayName;
    static final String WINDOWS_CATEGORY_ATTRIBUTES = "Attributes";

    /**
     * Constructs a query attribute command.
     * @param displayName the dialog name
     * @param name the command name
     * @param attributeName the name of the attribute to be changed
     * @param type the attribute type (class)
     */
    public QueryAttributeCommand(String displayName, String name,
                                 String attributeName, Class<?> type) {
        super(name);
        // this.fEditor = editor;
        this.fAttribute = attributeName;
        this.displayName = displayName;
        this.type = type;
    }

    private Dialog getDialog() {
        // i must create a new dialog every time because
        // the gui might have been closed... (?)
        JFrame parent = DrawPlugin.getGui().getFrame();
        final Dialog dialog = new Dialog(parent, "Change " + displayName + ":");
        input = new TextField();
        input.addKeyListener(new KeyAdapter() {
                public void keyPressed(KeyEvent e) {
                    if (e.getKeyCode() == KeyEvent.VK_ENTER) {
                        apply(type);
                    } else if (e.getKeyCode() == KeyEvent.VK_ESCAPE) {
                        closeDialog(dialog);
                    }
                }
            });
        dialog.add(input, "Center");
        Button apply = new Button(" Apply ");
        apply.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    apply(type);
                }
            });

        Button update = new Button(" Update ");
        update.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    execute();
                }
            });

        Button ok = new Button(" OK ");
        ok.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    apply(type);
                    closeDialog(dialog);
                }
            });

        Button cancel = new Button(" Cancel ");
        cancel.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    resetFocus();
                    closeDialog(dialog);
                }
            });

        Panel panel = new Panel();
        panel.setLayout(new FlowLayout());
        panel.add(apply);
        panel.add(update);
        panel.add(ok);
        panel.add(cancel);
        dialog.add(panel, "South");
        dialog.addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent e) {
                    closeDialog(dialog);
                }
            });
        dialog.pack();
        return dialog;
    }

    static void closeDialog(final Dialog dialog) {
        MenuManager.getInstance().getWindowsMenu().removeDialog(dialog);
        dialog.setVisible(false);
    }

    protected DrawingEditor getEditor() {
        DrawPlugin plugin = DrawPlugin.getCurrent();
        return (plugin == null) ? NullDrawingEditor.INSTANCE
                                : plugin.getDrawingEditor();
    }

    public void execute() {
        DrawingView view = getEditor().view();
        Dialog dialog = getDialog();
        Object value = null;
        FigureEnumeration k = view.selectionElements();
        while (k.hasMoreElements()) {
            Figure f = k.nextFigure();
            Object val = f.getAttribute(fAttribute);
            if (val != null) {
                if (value != null && !value.equals(val)) {
                    // different values, use previous value
                    value = null;
                    break;
                }
                value = val;
            }
        }
        String valuestr = "";
        if (value != null) {
            if (value instanceof Color) {
                valuestr = "0x"
                           + Integer.toHexString(((Color) value).getRGB())
                                    .substring(2);
            } else {
                valuestr = value.toString();
            }
        }
        input.setText(valuestr);
        resetFocus();
        dialog.setVisible(true);
        MenuManager.getInstance().getWindowsMenu()
                   .addDialog(WINDOWS_CATEGORY_ATTRIBUTES, dialog);
    }

    private void resetFocus() {
        input.requestFocus();
        input.setSelectionStart(0);
        input.setSelectionEnd(input.getText().length());
    }

    public boolean isExecutable() {
        if (getEditor() == NullDrawingEditor.INSTANCE) {
            return false;
        }
        return getEditor().view().selectionCount() > 0;
    }

    private boolean apply(Class<?> type) {
        String valuestr = input.getText();
        Object value = null;
        try {
            if (type == Integer.class) {
                value = Integer.valueOf(valuestr);
            } else if (type == String.class) {
                value = valuestr;
            } else if (type == Color.class) {
                value = Color.decode(valuestr);
            }
        } catch (NumberFormatException ex) {
        }
        if (value != null) {
            new ChangeAttributeCommand("", fAttribute, value).execute();
        }
        resetFocus();
        return value != null;
    }

    public Class<?> getType() {
        return this.type;
    }

    public String getAttributeName() {
        return this.fAttribute;
    }
}