package CH.ifa.draw.standard;

import CH.ifa.draw.DrawPlugin;

import CH.ifa.draw.application.MenuManager;

import CH.ifa.draw.framework.DrawingView;
import CH.ifa.draw.framework.Figure;
import CH.ifa.draw.framework.FigureEnumeration;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.JButton;
import javax.swing.JColorChooser;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JPanel;


/**
 * Command to change the color of a figure.
 * <p>
 * The command's effects are undoable step by step.
 * Each use of the apply button can be undone separately.
 * So this command doesn't need to inherit UndoableCommand.
 * </p>
 * @author Sven Offermann
 */
public class ChooseColorCommand extends QueryAttributeCommand {
    private JFrame lastParent = null;
    private JDialog dialog = null;
    private JColorChooser chooser;

    /**
     * Constructs a dialog with a JColorChooser to
     * change the color of figures, fonts etc.
     *
     * @param displayName the dialog name
     * @param name the command name
     * @param attributeName the name of the attribute to be changed
     * @param type the attribute type (class)
     */
    public ChooseColorCommand(String displayName, String name,
                              String attributeName, Class<?> type) {
        super(displayName, name, attributeName, type);
    }

    private JDialog getDialog() {
        // We can reuse an existing dialog, but we must create a new dialog
        // when the gui frame has been closed and recreated inbetween.
        // This is possible because the menu command objects are kept across
        // gui instances by the MenuManager
        // TODO: Redesign of the MenuManager or apply the same mechanism to
        //       all other editor dialogs (see bug #99).
        JFrame parent = DrawPlugin.getGui().getFrame();
        if (dialog != null) {
            if (lastParent == parent) {
                // reuse the existing dialog
                return dialog;
            } else {
                // or release the old one
                dialog.dispose();
            }
        }
        dialog = new JDialog(parent, "Change " + displayName + ":");
        lastParent = parent;

        chooser = new JColorChooser();
        chooser.setPreviewPanel(new JPanel());
        dialog.getContentPane().add(chooser, BorderLayout.CENTER);
        dialog.addKeyListener(new KeyAdapter() {
                public void keyPressed(KeyEvent e) {
                    if (e.getKeyCode() == KeyEvent.VK_ENTER) {
                        apply(getType());
                    } else if (e.getKeyCode() == KeyEvent.VK_ESCAPE) {
                        QueryAttributeCommand.closeDialog(dialog);
                    }
                }
            });

        JButton apply = new JButton(" Apply ");
        apply.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    apply(getType());
                }
            });

        JButton update = new JButton(" Update ");
        update.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    updateFromFigure();
                }
            });

        JButton ok = new JButton(" OK ");
        ok.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    apply(getType());
                    QueryAttributeCommand.closeDialog(dialog);
                }
            });

        JButton cancel = new JButton(" Cancel ");
        cancel.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    QueryAttributeCommand.closeDialog(dialog);
                }
            });

        JPanel panel = new JPanel();
        panel.setLayout(new FlowLayout());
        panel.add(apply);
        panel.add(update);
        panel.add(ok);
        panel.add(cancel);
        dialog.getContentPane().add(panel, BorderLayout.PAGE_END);
        dialog.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
        dialog.addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent e) {
                    QueryAttributeCommand.closeDialog(dialog);
                }
            });
        dialog.pack();
        return dialog;
    }

    public void execute() {
        JDialog dialog = getDialog();

        updateFromFigure();

        dialog.setVisible(true);
        MenuManager.getInstance().getWindowsMenu()
                   .addDialog(QueryAttributeCommand.WINDOWS_CATEGORY_ATTRIBUTES,
                              dialog);
    }

    private void updateFromFigure() {
        DrawingView view = getEditor().view();
        Object value = null;
        FigureEnumeration k = view.selectionElements();
        while (k.hasMoreElements()) {
            Figure f = k.nextFigure();
            Object val = f.getAttribute(fAttribute);
            if (val != null) {
                if (value != null && !value.equals(val)) {
                    // different values, use previous value
                    value = null;
                    break;
                }
                value = val;
            }
        }

        if (value != null) {
            if (value instanceof Color) {
                chooser.setColor((Color) value);
            }
        }
    }

    public boolean isExecutable() {
        if (getEditor() == NullDrawingEditor.INSTANCE) {
            return false;
        }
        return getEditor().view().selectionCount() > 0;
    }

    private boolean apply(Class<?> type) {
        Object value = null;
        if (type == Color.class) {
            value = chooser.getColor();
        }
        if (value != null) {
            new ChangeAttributeCommand("", fAttribute, value).execute();
        }
        return value != null;
    }
}