/*
 * Created on Apr 13, 2003
 */
package CH.ifa.draw.io;

import java.io.File;

import javax.swing.filechooser.FileFilter;


/**
 * SimpleFileFilter is the (abstract) class that is the basis for all FileFilters that
 * can be connected to the file types of the Drawings.
 * <p>
 * The SimpleFileFilter is used by the JFileChooser to determine the files that are displayed.
 * Implementations of SimpleFileFilter provide the extension and the description of the
 * denoted filetype. To define FileFilter that handle multiple extensions see
 * CombinationFileFilter.
 * </p>
 *
 * @author Lawrence Cabac
 */
public abstract class SimpleFileFilter extends FileFilter implements java.io.FileFilter {
    private String ext;
    private String description;
    private boolean allowDirectory = true;
    private boolean allowHidden = false;

    public void allowHidden(boolean b) {
        allowHidden = b;
    }

    public void allowDirectory(boolean b) {
        allowDirectory = b;
    }

    public boolean isHiddenAllowed() {
        return allowHidden;
    }

    public boolean isDirectoryAllowed() {
        return allowDirectory;
    }

    /**
    * A file is accepted by a SimpleFileFilter when its extension equals the extension
    * of the Simple|FileFilter.
    * @return true for a file that is accepted by the FileFileter.
    * @see javax.swing.filechooser.FileFilter#accept(java.io.File)
    */
    public boolean accept(File f) {
        if (f != null
                    && (allowHidden
                               || !f.isHidden() && !f.getName().startsWith("."))) {
            if (f.isDirectory()) {
                return allowDirectory;
            }

            String extension = getExtension(f);

            // logger.debug(extension +"  "+ ext +"  "+ (ext.equals(extension)));
            if (extension != null && ext.equals(extension)) {
                return true;
            }
        }
        return false;
    }

    /**
      * Returns the extension of the file's name .
      *
      * @see #getExtension
      * @see FileFilter#accept
      */
    protected String getExtension(File f) {
        if (f != null) {
            String filename = f.getName();
            int i = filename.lastIndexOf('.');
            if (i > 0 && i < filename.length() - 1) {
                return filename.substring(i + 1).toLowerCase();
            }
            ;
        }
        return null;
    }

    /**
     * The description of a file type is a human readable identifier that is
     * used in the select box of the JFileCooser.
     * @return The Description of the file type.
     * @see javax.swing.JFileChooser
     * @see javax.swing.filechooser.FileFilter#getDescription()
     */
    public String getDescription() {
        return description;
    }

    /**
     * @return The extension of the file type.
     */
    public String getExtension() {
        return ext;
    }

    public void setDescription(String string) {
        description = string;
    }

    public void setExtension(String string) {
        ext = string;
    }

    /**
     * Two SimpleFileFilter are equal when the extensions of them are equal.
         * @param ff The SimpleFileFilter that is compared.
         * @return true if the extensions of the two SimpleFileFilter are equal.
         */
    public boolean equals(SimpleFileFilter ff) {
        return ff != null && ff.getExtension().equals(getExtension());
    }
}