package de.renew.simulatorontology.shadow;

import java.io.Serial;

/**
 * Abstract class for nodes in {@link ShadowNet} instances, which are implemented
 * by {@link ShadowPlace} and {@link ShadowTransition}.
 * <p>
 * This class combines the common fields and methods for handling names, tracing
 * and getting the net the shadow node belongs to.
 */
public abstract class ShadowNode extends ShadowInscribable {
    /* Serialization support. */
    @Serial
    private static final long serialVersionUID = 6077849783470486673L;
    /** The name of this {@code ShadowNode}. */
    private String _name;
    /** {@code true}, if tracing is desired, which is the default. */
    private boolean _trace;
    /** The comment for this {@code ShadowNode} or {@code null} if it hasn't been set. */
    private String _comment;

    /**
     * Creates a new {@code ShadowNode} and registers it with the given net.
     * <p>
     * Also enables tracing, which can be turned off via {@link #setTrace(boolean)}.
     *
     * @param shadowNet the net this {@code ShadowNode} belongs to
     */
    public ShadowNode(ShadowNet shadowNet) {
        super(shadowNet);
        _name = null;
        setTrace(true);


        // Register myself with the net.
        shadowNet.add(this);
    }

    /**
     * Sets the name of this {@code ShadowNode}.
     *
     * @param name the given name
     */
    public void setName(String name) {
        _name = name;
    }

    /**
     * Returns the name of this {@code ShadowNode}.
     *
     * @return the name of this {@code ShadowNode}
     */
    public String getName() {
        return _name;
    }

    /**
     * Switches the trace flag on or off.
     *
     * @param trace {@code true}, if tracing is desired, otherwise {@code false}
     */
    public void setTrace(boolean trace) {
        if (_trace != trace) {
            _trace = trace;
        }
    }

    /**
     * Returns whether this shadow node is being traced.
     *
     * @return {@code true}, if trace flag is set, otherwise {@code false}
     */
    public boolean getTrace() {
        return _trace;
    }

    @Override
    public String toString() {
        String cln = getClass().getName();
        int ind = cln.lastIndexOf('.') + 1;
        if (ind > 0) {
            cln = cln.substring(ind);
        }
        if (_name == null) {
            return cln + " (" + getID() + ")";
        } else {
            return cln + " \"" + _name + "\"";
        }
    }

    /**
     * Sets the comment for this {@code ShadowNode}. Overwrites any previous comment.
     *
     * @param comment the comment to give this {@code ShadowNode}
     */
    public void setComment(String comment) {
        _comment = comment;

    }

    /**
     * Returns the comment for this {@code ShadowNode} or {@code null}
     * if it hasn't been set.
     *
     * @return the comment for this {@code ShadowNode} or {@code null}
     *         if it hasn't been set
     */
    public String getComment() {
        return _comment;
    }
}