package de.renew.database;

import de.renew.net.NetElementID;
import de.renew.net.SimulatablePlaceInstance;


/**
 * This class represents an action involving a token in a simulatable place.
 */
public class TokenAction {
    private final SimulatablePlaceInstance _placeInstance;
    private final Object _token;
    private final String _tokenId;
    private final double _time;

    TokenAction(SimulatablePlaceInstance placeInstance, Object token, double time) {
        _placeInstance = placeInstance;
        _token = token;
        _time = time;

        // Precalculate token ID. It will be necessary anyway.
        _tokenId = placeInstance.getTokenID(token);
    }

    /**
     * This method returns the placeInstance of the current instance.
     * @return the placeInstance
     */
    public SimulatablePlaceInstance getPlaceInstance() {
        return _placeInstance;
    }

    /**
     * Returns the moved token.
     * {@link java.lang.Object}
     * {@link de.renew.util.Value}
     *
     * @return the token, possibly wrapped in a <code>Value</code> object.
     */
    public Object getToken() {
        return _token;
    }

    /**
     * This method returns the TokenID of the current instance as a string.
     * @return the tokenID as a string
     */
    public String getTokenID() {
        return _tokenId;
    }

    /**
     * This method returns the NetID of the NetInstance place.
     * @return the id of the netInstance
     */
    public String getNetID() {
        return _placeInstance.getNetInstance().getID();
    }

    /**
     * This method returns the PlaceID of the current place instance.
     * @return the id of the placeInstance
     */
    public NetElementID getPlaceID() {
        return _placeInstance.getPlace().getID();
    }

    /**
     * This method returns the place's name of the current instance.
     * @return the name of the place of the place represented by the placeInstance
     */
    public String getPlaceName() {
        return _placeInstance.getPlace().toString();
    }

    /**
     * Returns the time stamp of the moved token.
     *
     * @return a <code>double</code> valued time stamp
     */
    public double getTime() {
        return _time;
    }
}